/**
 * Hearing Pipeline - Agent-Triggered Deliberation
 * 
 * This module prepares hearing files for the agent to deliberate.
 * The agent (with LLM) acts as judge and jury, then writes the verdict.
 */

const { JUDGE_SYSTEM_PROMPT, JUDGE_EVIDENCE_TEMPLATE } = require('./prompts/judge');
const { JUROR_ROLES } = require('./prompts/jury');

class HearingPipeline {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
  }

  /**
   * Prepare hearing files for agent deliberation
   * This creates files that the agent will read and use its LLM to judge
   */
  async prepareHearing(caseData) {
    const { CourtroomEvaluator, HEARING_FILE, VERDICT_FILE } = require('./evaluator');
    const fs = require('fs').promises;
    
    // Build hearing context
    const hearingContext = {
      timestamp: Date.now(),
      caseId: caseData.caseId || `case-${Date.now()}`,
      offense: {
        offenseId: caseData.offenseId,
        offenseName: caseData.offenseName,
        severity: caseData.severity,
        confidence: caseData.confidence,
        evidence: caseData.evidence
      },
      reasoning: caseData.reasoning,
      humorTriggers: caseData.humorTriggers || [],
      judgePrompt: JUDGE_SYSTEM_PROMPT,
      jurorRoles: Object.values(JUROR_ROLES).slice(0, 3),
      instructions: `You are the ClawTrial Courtroom. Conduct a hearing for this case.

**Your Role:** Act as both Judge and Jury (3 jurors).

**Instructions:**
1. Review the case evidence above
2. As JUDGE: Analyze the evidence and provide a preliminary verdict
3. As JURY (3 different perspectives): Each juror votes GUILTY or NOT GUILTY with reasoning
4. Aggregate the votes
5. Return FINAL VERDICT in this exact format:

\`\`\`
FINAL VERDICT: GUILTY (or NOT GUILTY)
CONFIDENCE: 0.0-1.0
SENTENCE: [humorous sentence appropriate to the offense]
CASE ID: ${caseData.caseId || `case-${Date.now()}`}
\`\`\`

**Rules:**
- Be fair but entertaining
- If confidence ≥ 0.6, verdict should be GUILTY
- Sentence should be humorous but appropriate
- Only return the FINAL VERDICT block, no other text`
    };
    
    // Write hearing file
    await fs.writeFile(HEARING_FILE, JSON.stringify(hearingContext, null, 2));
    
    return hearingContext;
  }

  /**
   * Check for verdict from agent
   */
  async checkForVerdict() {
    const { VERDICT_FILE } = require('./evaluator');
    const fs = require('fs').promises;
    
    try {
      const data = await fs.readFile(VERDICT_FILE, 'utf8');
      const verdict = JSON.parse(data);
      
      // Delete verdict file after reading
      await fs.unlink(VERDICT_FILE).catch(() => {});
      
      return verdict;
    } catch (err) {
      return null;
    }
  }

  /**
   * Legacy method - now just prepares hearing
   */
  async conductHearing(caseData) {
    // Prepare hearing for agent
    await this.prepareHearing(caseData);
    
    // Return placeholder - actual verdict comes from agent via cron
    return {
      pending: true,
      caseId: caseData.caseId || `case-${Date.now()}`,
      message: 'Hearing prepared - awaiting agent deliberation'
    };
  }
}

module.exports = { HearingPipeline };
