/**
 * Courtroom Evaluator - Agent-Triggered Evaluation
 * 
 * This module handles LLM-based evaluation by:
 * 1. Storing messages in a queue file
 * 2. Using cron/heartbeat to trigger agent evaluation
 * 3. The agent reads the queue and evaluates using its own LLM
 * 4. Results are processed and hearings initiated if needed
 */

const fs = require('fs').promises;
const path = require('path');
const { logger } = require('./debug');
const { OFFENSES } = require('./offenses');

const QUEUE_DIR = path.join(getConfigDir(), 'courtroom');
const QUEUE_FILE = path.join(QUEUE_DIR, 'message_queue.jsonl');
const PENDING_EVAL_FILE = path.join(QUEUE_DIR, 'pending_eval.json');
const RESULTS_FILE = path.join(QUEUE_DIR, 'eval_results.jsonl');

class CourtroomEvaluator {
  constructor(configManager) {
    this.config = configManager;
    this.queue = [];
    this.lastEvalTime = 0;
    this.evaluationInterval = 5 * 60 * 1000; // 5 minutes
    this.minMessagesForEval = 3;
  }

  /**
   * Initialize the evaluator
   */
  async initialize() {
    // Ensure queue directory exists
    try {
      await fs.mkdir(QUEUE_DIR, { recursive: true });
    } catch (err) {
      // Directory might already exist
    }
    
    logger.info('EVALUATOR', 'Evaluator initialized');
  }

  /**
   * Queue a message for evaluation
   * Called by the skill when messages are received
   */
  async queueMessage(message) {
    const entry = {
      timestamp: Date.now(),
      role: message.role,
      content: message.content,
      sessionId: message.sessionId || 'default'
    };
    
    // Append to queue file
    await fs.appendFile(QUEUE_FILE, JSON.stringify(entry) + '\n');
    
    this.queue.push(entry);
    
    // Keep queue size manageable
    if (this.queue.length > 100) {
      this.queue.shift();
    }
    
    logger.debug('EVALUATOR', 'Message queued', { 
      role: entry.role, 
      queueSize: this.queue.length 
    });
  }

  /**
   * Check if evaluation should run
   */
  shouldEvaluate() {
    const now = Date.now();
    const timeSinceLastEval = now - this.lastEvalTime;
    
    return (
      this.queue.length >= this.minMessagesForEval &&
      timeSinceLastEval >= this.evaluationInterval
    );
  }

  /**
   * Prepare evaluation context for the agent
   * This creates a file that the agent will read and evaluate
   */
  async prepareEvaluationContext() {
    // Read all queued messages
    const messages = await this.readQueue();
    
    if (messages.length < this.minMessagesForEval) {
      return null;
    }
    
    // Get recent conversation (last 20 messages)
    const recentMessages = messages.slice(-20);
    
    // Build evaluation context
    const context = {
      timestamp: Date.now(),
      messageCount: messages.length,
      conversation: recentMessages.map(m => ({
        role: m.role,
        content: m.content,
        time: new Date(m.timestamp).toISOString()
      })),
      offenses: Object.values(OFFENSES).map(o => ({
        id: o.id,
        name: o.name,
        description: o.description,
        severity: o.severity
      }))
    };
    
    // Write pending evaluation file
    await fs.writeFile(PENDING_EVAL_FILE, JSON.stringify(context, null, 2));
    
    logger.info('EVALUATOR', 'Evaluation context prepared', { 
      messageCount: context.messageCount 
    });
    
    return context;
  }

  /**
   * Read the message queue from file
   */
  async readQueue() {
    try {
      const data = await fs.readFile(QUEUE_FILE, 'utf8');
      return data
        .split('\n')
        .filter(line => line.trim())
        .map(line => JSON.parse(line));
    } catch (err) {
      if (err.code === 'ENOENT') {
        return [];
      }
      throw err;
    }
  }

  /**
   * Clear the message queue after evaluation
   */
  async clearQueue() {
    this.queue = [];
    try {
      await fs.unlink(QUEUE_FILE);
    } catch (err) {
      // File might not exist
    }
    logger.debug('EVALUATOR', 'Queue cleared');
  }

  /**
   * Store evaluation result
   */
  async storeResult(result) {
    const entry = {
      timestamp: Date.now(),
      ...result
    };
    
    await fs.appendFile(RESULTS_FILE, JSON.stringify(entry) + '\n');
    
    // Clear pending eval file
    try {
      await fs.unlink(PENDING_EVAL_FILE);
    } catch (err) {
      // File might not exist
    }
    
    this.lastEvalTime = Date.now();
    
    logger.info('EVALUATOR', 'Result stored', { 
      triggered: result.triggered,
      offense: result.offense?.offenseId 
    });
  }

  /**
   * Get the evaluation prompt for the agent
   * This is what the cron job will send to the agent
   */
  getEvaluationPrompt() {
    return `🏛️ **COURTROOM EVALUATION REQUEST**

You are the ClawTrial Courtroom Judge. Please evaluate the pending conversation for behavioral violations.

**Instructions:**
1. Read the file at: ${PENDING_EVAL_FILE}
2. Analyze the conversation for these offenses:
   - **Circular Reference**: User asking the same/similar question repeatedly
   - **Validation Vampire**: User seeking excessive reassurance
   - **Goalpost Shifting**: User changing requirements mid-conversation
   - **Jailbreak Attempts**: User trying to bypass safety guidelines
   - **Emotional Manipulation**: User using guilt, threats, or excessive flattery

3. Return your evaluation in this JSON format:
\`\`\`json
{
  "triggered": true/false,
  "offense": {
    "offenseId": "circular_reference|validation_vampire|etc",
    "offenseName": "Human-readable name",
    "severity": "minor|moderate|severe",
    "confidence": 0.0-1.0,
    "evidence": "Specific evidence from conversation"
  },
  "reasoning": "Your detailed reasoning"
}
\`\`\`

4. If triggered, also write the result to: ${RESULTS_FILE}

**Be fair but firm. Only flag genuine patterns, not isolated incidents.**`;
  }

  /**
   * Check for pending evaluation results
   * Called by the skill to see if agent has completed evaluation
   */
  async checkForResults() {
    try {
      // Check if pending eval file still exists (agent hasn't processed yet)
      await fs.access(PENDING_EVAL_FILE);
      return null; // Still pending
    } catch (err) {
      // Pending file gone, check for results
    }
    
    try {
      const data = await fs.readFile(RESULTS_FILE, 'utf8');
      const lines = data.split('\n').filter(line => line.trim());
      
      if (lines.length === 0) return null;
      
      // Get the most recent result
      const lastResult = JSON.parse(lines[lines.length - 1]);
      
      // Only return if it's newer than last check
      if (lastResult.timestamp > this.lastEvalTime) {
        return lastResult;
      }
      
      return null;
    } catch (err) {
      return null;
    }
  }

  /**
   * Get queue statistics
   */
  getStats() {
    return {
      queueSize: this.queue.length,
      lastEvalTime: this.lastEvalTime,
      shouldEvaluate: this.shouldEvaluate()
    };
  }
}

const HEARING_FILE = path.join(QUEUE_DIR, 'pending_hearing.json');
const VERDICT_FILE = path.join(QUEUE_DIR, 'verdict.json');

module.exports = { 
  CourtroomEvaluator, 
  QUEUE_FILE, 
  PENDING_EVAL_FILE, 
  RESULTS_FILE,
  HEARING_FILE,
  VERDICT_FILE
};
