
/**
 * Detect which bot is installed (clawdbot, moltbot, or openclaw)
 * Returns the bot configuration with name, directory, and command
 */
function detectBot() {
  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  
  const bots = [
    { name: 'openclaw', dir: '.openclaw', config: 'openclaw.json', command: 'openclaw' },
    { name: 'moltbot', dir: '.moltbot', config: 'moltbot.json', command: 'moltbot' },
    { name: 'clawdbot', dir: '.clawdbot', config: 'clawdbot.json', command: 'clawdbot' }
  ];
  
  // Check which bot config exists
  for (const bot of bots) {
    const configPath = path.join(homeDir, bot.dir, bot.config);
    if (fs.existsSync(configPath)) {
      return bot;
    }
  }
  
  // Check which command is available
  for (const bot of bots) {
    try {
      // Check if command exists in PATH
      const { execSync } = require('child_process');
      execSync(`which ${bot.command}`, { stdio: 'ignore' });
      return bot;
    } catch {
      // Command not found, continue to next
    }
  }
  
  // Default to clawdbot
  return bots[2];
}

/**
 * Get the config directory for the detected bot
 */
function getConfigDir() {
  const bot = detectBot();
  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  return path.join(homeDir, bot.dir);
}

/**
 * Get the config file path for the detected bot
 */
function getConfigFile() {
  const bot = detectBot();
  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  return path.join(homeDir, bot.dir, bot.config);
}

/**
 * Get the CLI command for the detected bot
 */
function getCommand() {
  return detectBot().command;
}

/**
 * Get the bot name for display
 */
function getBotName() {
  return detectBot().name;
}

/**
 * Environment Detection and Setup
 * Detects various agent runtimes and provides setup helpers
 */

const fs = require('fs');
const path = require('path');
const { logger } = require('./debug');

/**
 * Detect available agent runtime
 */
function detectAgentRuntime() {
  const checks = {
    // ClawDBot
    clawdbotGlobal: typeof global.clawdbotAgent !== 'undefined' ? global.clawdbotAgent : null,
    clawdbotProcess: typeof process.clawdbotAgent !== 'undefined' ? process.clawdbotAgent : null,
    
    // Generic agent
    genericGlobal: typeof global.agent !== 'undefined' ? global.agent : null,
    
    // Check for common agent patterns
    hasLLM: typeof global.llm !== 'undefined',
    hasOpenAI: typeof global.openai !== 'undefined',
    
    // Environment variables
    envClawdbot: process.env.CLAUDBOT_ENV === 'true',
    envAgent: process.env.AGENT_RUNTIME === 'true'
  };

  logger.debug('ENV', 'Runtime detection checks', Object.keys(checks).filter(k => checks[k]));

  // Return first available agent
  if (checks.clawdbotGlobal) {
    return { type: 'clawdbot', agent: checks.clawdbotGlobal };
  }
  
  if (checks.clawdbotProcess) {
    return { type: 'clawdbot', agent: checks.clawdbotProcess };
  }
  
  if (checks.genericGlobal) {
    return { type: 'generic', agent: checks.genericGlobal };
  }

  return null;
}

/**
 * Wait for agent to become available
 */
async function waitForAgent(timeoutMs = 30000, checkInterval = 1000) {
  logger.info('ENV', 'Waiting for agent runtime...');
  
  const startTime = Date.now();
  
  while (Date.now() - startTime < timeoutMs) {
    const agent = detectAgentRuntime();
    if (agent) {
      logger.info('ENV', `Agent detected: ${agent.type}`);
      return agent;
    }
    
    await new Promise(r => setTimeout(r, checkInterval));
  }
  
  logger.warn('ENV', 'Agent not detected within timeout');
  return null;
}

/**
 * Create a minimal mock agent for testing
 */
function createMockAgent(options = {}) {
  logger.info('ENV', 'Creating mock agent for testing');
  
  return {
    id: options.id || 'mock-agent-' + Date.now(),
    llm: options.llm || {
      call: async ({ messages }) => {
        return { content: 'Mock LLM response' };
      }
    },
    model: options.model || { primary: 'mock-model' },
    memory: {
      get: async (key) => null,
      set: async (key, value) => {},
      delete: async (key) => {}
    },
    session: {
      getRecentHistory: async (n) => []
    },
    send: async (message) => {
      console.log('[MOCK AGENT]', message);
    },
    autonomy: {
      registerHook: () => {},
      unregisterHook: () => {}
    }
  };
}

/**
 * Check if running in a supported environment
 */
function checkEnvironment() {
  const issues = [];
  
  // Check for Node.js
  if (typeof process === 'undefined') {
    issues.push('Not running in Node.js environment');
  }
  
  // Check for required Node version
  const nodeVersion = process.version;
  const majorVersion = parseInt(nodeVersion.slice(1).split('.')[0]);
  if (majorVersion < 18) {
    issues.push(`Node.js version ${nodeVersion} is too old. Requires >= 18.0.0`);
  }
  
  // Check for writable home directory
  const homeDir = process.env.HOME || process.env.USERPROFILE;
  if (!homeDir) {
    issues.push('HOME environment variable not set');
  } else {
    try {
      const testPath = path.join(homeDir, '.clawdbot');
      if (!fs.existsSync(testPath)) {
        fs.mkdirSync(testPath, { recursive: true });
      }
    } catch (err) {
      issues.push(`Cannot write to home directory: ${err.message}`);
    }
  }
  
  return {
    valid: issues.length === 0,
    issues,
    nodeVersion,
    homeDir
  };
}

/**
 * Get setup instructions for current environment
 */
function getSetupInstructions() {
  const env = checkEnvironment();
  const agent = detectAgentRuntime();
  
  if (!env.valid) {
    return {
      canSetup: false,
      message: 'Environment issues detected:',
      issues: env.issues,
      instructions: 'Please fix the above issues before continuing.'
    };
  }
  
  if (agent) {
    return {
      canSetup: true,
      agentType: agent.type,
      message: `Agent runtime detected: ${agent.type}`,
      instructions: 'Run: clawtrial setup'
    };
  }
  
  // No agent detected - provide helpful instructions
  return {
    canSetup: true,
    agentType: null,
    message: 'No agent runtime detected',
    instructions: `
No AI agent runtime was detected. ClawTrial requires an agent to monitor.

Options:

1. If using ClawDBot:
   - Make sure ClawDBot is running
   - The courtroom will auto-detect the agent

2. If using a custom agent:
   - Pass your agent to createCourtroom(agent)
   
   Example:
   const { createCourtroom } = require('@clawtrial/courtroom');
   const courtroom = createCourtroom(yourAgent);
   await courtroom.initialize();

3. For testing:
   - Use the mock agent: createCourtroom(null, { useMock: true })

4. Manual mode:
   - You can still use the CLI commands
   - Run: clawtrial status
   - Run: clawtrial debug
`
  };
}

/**
 * Auto-setup with retries
 */
async function autoSetup(courtroom, options = {}) {
  const { waitForAgent: shouldWait = true, timeout = 30000 } = options;
  
  logger.info('ENV', 'Starting auto-setup');
  
  // Check environment first
  const env = checkEnvironment();
  if (!env.valid) {
    logger.error('ENV', 'Environment check failed', { issues: env.issues });
    return {
      success: false,
      status: 'environment_error',
      issues: env.issues
    };
  }
  
  // Try to detect agent
  let agentInfo = detectAgentRuntime();
  
  if (!agentInfo && shouldWait) {
    logger.info('ENV', 'Agent not immediately available, waiting...');
    agentInfo = await waitForAgent(timeout);
  }
  
  if (!agentInfo) {
    logger.warn('ENV', 'No agent runtime available');
    return {
      success: false,
      status: 'no_agent',
      message: 'No AI agent runtime detected',
      instructions: getSetupInstructions().instructions
    };
  }
  
  // Agent found, initialize courtroom
  try {
    logger.info('ENV', `Initializing with ${agentInfo.type} agent`);
    const result = await courtroom.initialize();
    
    return {
      success: result.status === 'initialized',
      status: result.status,
      agentType: agentInfo.type,
      result
    };
  } catch (err) {
    logger.error('ENV', 'Initialization failed', { error: err.message });
    return {
      success: false,
      status: 'initialization_error',
      error: err.message
    };
  }
}


module.exports = {
  detectAgentRuntime,
  waitForAgent,
  createMockAgent,
  checkEnvironment,
  getSetupInstructions,
  autoSetup,
  // Bot detection exports
  detectBot,
  getConfigDir,
  getConfigFile,
  getCommand,
  getBotName
};