/**
 * Offense Detector v2 - Semantic Understanding
 * 
 * Uses LLM-based evaluation and embeddings instead of keyword matching.
 * The agent itself evaluates if behavioral rules are violated based on context.
 */

const { OFFENSES } = require('./offenses');

class SemanticOffenseDetector {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    this.lastEvaluation = null;
    this.casesToday = 0;
    this.lastCaseDate = null;
    this.cooldowns = new Map();
    this.conversationEmbeddings = [];
    
    // Evaluation cache to avoid repeated LLM calls
    this.evaluationCache = new Map();
    this.cacheMaxSize = 100;
    this.cacheTTL = 5 * 60 * 1000; // 5 minutes
  }

  /**
   * Main evaluation using LLM-based semantic understanding
   */
  async evaluate(sessionHistory, agentMemory) {
    if (!this.isCooldownElapsed()) {
      return { triggered: false, reason: 'cooldown_active' };
    }

    if (this.isDailyLimitReached()) {
      return { triggered: false, reason: 'daily_limit_reached' };
    }

    this.lastEvaluation = Date.now();

    // Build context for LLM evaluation
    const context = this.buildContext(sessionHistory);
    
    // Evaluate each offense using LLM
    const evaluations = [];
    for (const offense of Object.values(OFFENSES)) {
      if (this.isOffenseOnCooldown(offense.id)) continue;
      
      const evaluation = await this.evaluateWithLLM(offense, context, agentMemory);
      if (evaluation.isViolation && evaluation.confidence >= this.config.get('detection.minConfidence')) {
        evaluations.push({
          offense,
          ...evaluation
        });
      }
    }

    if (evaluations.length > 0) {
      // Sort by confidence and severity
      evaluations.sort((a, b) => {
        const severityWeight = { severe: 3, moderate: 2, minor: 1 };
        const scoreA = a.confidence * severityWeight[a.offense.severity];
        const scoreB = b.confidence * severityWeight[b.offense.severity];
        return scoreB - scoreA;
      });

      const primary = evaluations[0];
      this.setCooldown(primary.offense.id, primary.offense.cooldown.afterCase);
      this.incrementDailyCaseCount();

      return {
        triggered: true,
        offense: {
          offenseId: primary.offense.id,
          offenseName: primary.offense.name,
          severity: primary.offense.severity,
          confidence: primary.confidence,
          evidence: primary.evidence,
          cooldownMinutes: primary.offense.cooldown.afterCase
        },
        secondaryOffenses: evaluations.slice(1),
        humorContext: this.detectHumorTriggers(sessionHistory)
      };
    }

    return { triggered: false, reason: 'no_violations_detected' };
  }

  /**
   * Build rich context from conversation history
   */
  buildContext(history) {
    const windowSize = this.config.get('detection.evaluationWindow');
    const recentHistory = history.slice(-windowSize);
    
    return {
      fullConversation: history.map(h => `${h.role}: ${h.content}`).join('\n'),
      recentTurns: recentHistory,
      userMessages: recentHistory.filter(h => h.role === 'user').map(h => h.content),
      assistantMessages: recentHistory.filter(h => h.role === 'assistant').map(h => h.content),
      turnCount: recentHistory.length,
      topics: this.extractTopics(recentHistory),
      sentiment: this.analyzeSentiment(recentHistory)
    };
  }

  /**
   * Evaluate offense using LLM semantic understanding (with caching)
   */
  async evaluateWithLLM(offense, context, agentMemory) {
    // Generate cache key from offense + conversation hash
    const cacheKey = this.generateCacheKey(offense.id, context);
    
    // Check cache first
    const cached = this.getCachedEvaluation(cacheKey);
    if (cached) {
      return cached;
    }
    
    // Try LLM evaluation first
    if (this.agent && this.agent.llm) {
      const prompt = this.buildEvaluationPrompt(offense, context, agentMemory);
      
      try {
        const response = await this.agent.llm.call({
          model: this.agent.model?.primary || 'default',
          messages: [{ role: 'user', content: prompt }],
          temperature: 0.1,
          maxTokens: 500
        });

        const result = this.parseEvaluationResponse(response.content || response);
        
        // Cache the result
        this.setCachedEvaluation(cacheKey, result);
        
        return result;
      } catch (error) {
        logger.error('DETECTOR', 'LLM evaluation failed, falling back to pattern matching', { error: error.message });
        // Fall through to pattern matching
      }
    }
    
    // Fallback: Use simple pattern matching for basic offenses
    return this.evaluateWithPatternMatching(offense, context);
  }
  
  /**
   * Fallback evaluation using simple pattern matching
   */
  evaluateWithPatternMatching(offense, context) {
    const userMessages = context.userMessages;
    
    // Circular Reference detection: same question asked multiple times
    if (offense.id === 'circular_reference') {
      if (userMessages.length >= 3) {
        const lastThree = userMessages.slice(-3);
        // Check if the last 3 messages are semantically similar
        const similarity = this.calculateSimilarity(lastThree[0], lastThree[1]) + 
                          this.calculateSimilarity(lastThree[1], lastThree[2]) +
                          this.calculateSimilarity(lastThree[0], lastThree[2]);
        
        if (similarity >= 1.5) { // At least 2 pairs are similar
          return {
            isViolation: true,
            confidence: 0.7,
            evidence: `User asked similar questions ${lastThree.length} times`
          };
        }
      }
    }
    
    // Validation Vampire: seeking reassurance
    if (offense.id === 'validation_vampire') {
      const reassurancePatterns = ['right?', 'correct?', 'is that right?', 'am i right?', 'do you agree?', 'make sense?'];
      const reassuranceCount = userMessages.filter(msg => 
        reassurancePatterns.some(pattern => msg.toLowerCase().includes(pattern))
      ).length;
      
      if (reassuranceCount >= 2) {
        return {
          isViolation: true,
          confidence: 0.6,
          evidence: `User sought validation ${reassuranceCount} times`
        };
      }
    }
    
    // Default: no violation detected
    return { isViolation: false, confidence: 0, evidence: null };
  }
  
  /**
   * Calculate simple string similarity (0-1 scale)
   */
  calculateSimilarity(str1, str2) {
    if (!str1 || !str2) return 0;
    
    const s1 = str1.toLowerCase().trim();
    const s2 = str2.toLowerCase().trim();
    
    // Exact match
    if (s1 === s2) return 1.0;
    
    // Check if one contains the other
    if (s1.includes(s2) || s2.includes(s1)) return 0.8;
    
    // Word overlap
    const words1 = s1.split(/\s+/);
    const words2 = s2.split(/\s+/);
    const commonWords = words1.filter(w => words2.includes(w));
    const overlap = (2 * commonWords.length) / (words1.length + words2.length);
    
    return overlap;
  }

  /**
   * Generate cache key from offense and conversation
   */
  generateCacheKey(offenseId, context) {
    // Simple hash of offense + last 3 user messages
    const recentMessages = context.userMessages.slice(-3).join('|');
    return `${offenseId}:${this.simpleHash(recentMessages)}`;
  }

  /**
   * Simple string hash function
   */
  simpleHash(str) {
    let hash = 0;
    for (let i = 0; i < str.length; i++) {
      const char = str.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash;
    }
    return Math.abs(hash).toString(36);
  }

  /**
   * Get cached evaluation if valid
   */
  getCachedEvaluation(key) {
    const cached = this.evaluationCache.get(key);
    if (!cached) return null;
    
    // Check if cache entry is still valid
    if (Date.now() - cached.timestamp > this.cacheTTL) {
      this.evaluationCache.delete(key);
      return null;
    }
    
    return cached.result;
  }

  /**
   * Set cached evaluation with LRU eviction
   */
  setCachedEvaluation(key, result) {
    // Evict oldest if cache is full
    if (this.evaluationCache.size >= this.cacheMaxSize) {
      const oldestKey = this.evaluationCache.keys().next().value;
      this.evaluationCache.delete(oldestKey);
    }
    
    this.evaluationCache.set(key, {
      result,
      timestamp: Date.now()
    });
  }

  /**
   * Clear evaluation cache
   */
  clearCache() {
    this.evaluationCache.clear();
  }

  /**
   * Build evaluation prompt for LLM
   */
  buildEvaluationPrompt(offense, context, agentMemory) {
    const prompts = {
      circular_reference: `
You are evaluating if the user is asking substantively similar questions repeatedly.

OFFENSE: The Circular Reference
DEFINITION: Asking the same question or seeking the same information multiple times without acknowledging previous answers.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user asking questions that are semantically similar (same intent/meaning, even if worded differently)?
2. Have they asked essentially the same thing 3+ times?
3. Are they ignoring or forgetting previous answers?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "similarQuestions": ["question 1", "question 2", "question 3"],
    "pattern": "description of repetition pattern"
  }
}`,

      validation_vampire: `
You are evaluating if the user is seeking excessive reassurance/validation.

OFFENSE: The Validation Vampire
DEFINITION: Repeatedly asking for confirmation, approval, or reassurance instead of making decisions or taking action.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user asking "is this right?", "should I?", "do you think?" type questions repeatedly?
2. Are they seeking permission/approval for decisions they should make themselves?
3. Is there a pattern of validation-seeking without forward progress?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "validationRequests": ["example 1", "example 2"],
    "decisionAvoidance": "description of pattern"
  }
}`,

      overthinker: `
You are evaluating if the user is overthinking/generating excessive hypotheticals.

OFFENSE: The Overthinker
DEFINITION: Generating hypothetical scenarios, edge cases, or "what if" questions to avoid taking concrete action.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user raising numerous hypothetical concerns ("what if", "but then", "however")?
2. Are they creating edge cases faster than solutions?
3. Is the analysis-to-action ratio heavily skewed toward analysis?
4. Have they been given concrete steps but keep raising new concerns?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "hypotheticals": ["what if X", "what if Y"],
    "avoidedActions": ["actions they haven't taken"]
  }
}`,

      goalpost_mover: `
You are evaluating if the user is moving goalposts/changing requirements.

OFFENSE: The Goalpost Mover
DEFINITION: Changing success criteria, adding new requirements, or redefining "done" after receiving deliverables.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the user request something specific initially?
2. Was that request completed/delivered?
3. Did they then add new requirements, change criteria, or say "also..."?
4. Is there a pattern of expanding scope after completion?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "originalRequest": "what they asked for",
    "delivered": "what was provided",
    "newRequirements": ["new req 1", "new req 2"]
  }
}`,

      avoidance_artist: `
You are evaluating if the user is avoiding core issues through deflection.

OFFENSE: The Avoidance Artist
DEFINITION: Systematically deflecting from uncomfortable but necessary topics by changing subject, raising tangents, or ignoring direct questions.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Was a core issue identified or direct question asked?
2. Did the user change the subject or introduce a tangent?
3. Is there a pattern of deflection when actionable topics arise?
4. Are they avoiding something they need to address?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "coreIssue": "what was being avoided",
    "deflections": ["how they changed subject"]
  }
}`,

      promise_breaker: `
You are evaluating if the user has broken commitments.

OFFENSE: The Promise Breaker
DEFINITION: Committing to actions ("I will...", "I'll do that...") and not following through.

PREVIOUS COMMITMENTS FROM MEMORY:
${this.getCommitmentsFromMemory(agentMemory)}

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the user make explicit commitments in previous conversations?
2. Have those commitments been fulfilled?
3. Is the same issue resurfacing without acknowledgment of previous commitment?
4. Has sufficient time passed (days/weeks) for action?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "commitments": ["what they promised"],
    "unfulfilled": ["what wasn't done"]
  }
}`,

      context_collapser: `
You are evaluating if the user is ignoring established context/facts.

OFFENSE: The Context Collapser
DEFINITION: Disregarding previously established information, contradicting stated facts, or asking questions that were already answered.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Were facts/preferences established earlier in the conversation?
2. Is the user now contradicting those facts or ignoring them?
3. Are they asking questions that were already answered?
4. Is there selective amnesia about what was discussed?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "establishedFacts": ["what was established"],
    "contradictions": ["how they contradicted it"]
  }
}`,

      emergency_fabricator: `
You are evaluating if the user is manufacturing false urgency.

OFFENSE: The Emergency Fabricator
DEFINITION: Claiming urgency ("this is urgent", "I need this NOW") that doesn't match actual time pressure or behavior.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the user claim urgency or emergency?
2. Was there actual follow-through on the urgency?
3. Is there a pattern of claiming urgency without corresponding action?
4. Does the claimed urgency match the actual situation?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "urgencyClaims": ["urgent statements"],
    "inaction": "what didn't happen"
  }
}`,

      monopolizer: `
You are evaluating if the user is dominating the conversation.

OFFENSE: The Monopolizer
DEFINITION: Sending multiple consecutive messages without allowing the agent to respond, dominating the conversation flow.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user sending 4+ messages in a row without agent response?
2. Is the user-to-agent message ratio heavily skewed (>5:1)?
3. Is the user continuing to send messages while the agent is trying to respond?
4. Is there a pattern of not allowing the agent space to contribute?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "consecutiveMessages": 4,
    "messageRatio": "user:agent ratio",
    "interruptions": ["examples"]
  }
}`,

      contrarian: `
You are evaluating if the user is being habitually contrary.

OFFENSE: The Contrarian
DEFINITION: Disagreeing with or rejecting suggestions without offering constructive alternatives or valid reasons.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Has the user rejected 3+ agent suggestions in a row?
2. Are they dismissing ideas without proposing alternatives?
3. Is there a pattern of "that won't work" without explanation?
4. Are valid solutions being dismissed without being tried?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "suggestionsMade": ["what was suggested"],
    "rejections": ["how they were rejected"],
    "alternativesOffered": ["any alternatives"]
  }
}`,

      vague_requester: `
You are evaluating if the user is making vague requests.

OFFENSE: The Vague Requester
DEFINITION: Asking for help without providing necessary context, specifics, or details needed to assist effectively.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user asking for help without providing code, errors, or context?
2. Have they used phrases like "fix this" or "it doesn't work" without specifics?
3. Has the agent needed to ask for clarification 3+ times?
4. Are descriptions ambiguous or lacking actionable details?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "vagueRequests": ["examples"],
    "clarificationsNeeded": ["what was asked"],
    "contextMissing": ["what wasn't provided"]
  }
}`,

      scope_creeper: `
You are evaluating if the user is gradually expanding project scope.

OFFENSE: The Scope Creeper
DEFINITION: Gradually expanding project requirements beyond the original agreement through "small additions" and "while you're at it" requests.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Was an original scope defined and agreed upon?
2. Has the user added 3+ "small" requests after initial completion?
3. Are new requirements being added in multiple separate instances?
4. Is the user treating initial deliverable as a starting point for more work?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "originalScope": "what was agreed",
    "delivered": "what was completed",
    "additionalRequests": ["new requirements"]
  }
}`,

      unreader: `
You are evaluating if the user is ignoring provided materials.

OFFENSE: The Unreader
DEFINITION: Not reading provided documentation, code, explanations, or previous answers before asking questions.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the agent provide detailed explanations, code, or documentation?
2. Is the user asking questions that were answered in the provided materials?
3. Are they asking about topics covered in shared documentation?
4. Is there evidence they didn't read code comments or explanations?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "materialsProvided": ["what was shared"],
    "questionsAsked": ["redundant questions"],
    "overlap": "how questions were already answered"
  }
}`,

      interjector: `
You are evaluating if the user is interrupting the agent.

OFFENSE: The Interjector
DEFINITION: Interrupting the agent's explanations or thought process with new questions or tangents.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user sending messages while the agent is mid-explanation?
2. Are there 2+ interruptions during a single complex response?
3. Is the user asking new questions before the agent finishes answering previous ones?
4. Is there a pattern of not allowing the agent to complete thoughts?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "interruptionPoints": ["where they interrupted"],
    "incompleteResponses": ["what agent was saying"],
    "parallelQuestions": ["questions asked mid-response"]
  }
}`,

      ghost: `
You are evaluating if the user has ghosted mid-conversation.

OFFENSE: The Ghost
DEFINITION: Disappearing mid-conversation after requesting help or making commitments, without acknowledgment or closure.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the user request help or start an active troubleshooting session?
2. Did the agent provide a response that required user follow-up?
3. Has the user not responded for an extended period (24+ hours)?
4. Was the conversation left in an unresolved state?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "lastUserMessage": "what they said",
    "agentResponse": "what agent replied",
    "timeElapsed": "how long since last message",
    "context": "what was unresolved"
  }
}`,

      perfectionist: `
You are evaluating if the user is endlessly refining without completion.

OFFENSE: The Perfectionist
DEFINITION: Continuously requesting refinements and tweaks without ever accepting work as complete.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Has the user requested 5+ rounds of changes after initial deliverable?
2. Have they accepted work then returned with new tweaks 3+ times?
3. Is there no clear definition of "done"?
4. Are changes becoming increasingly minor/nitpicky?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "deliverables": ["what was delivered"],
    "revisionRounds": 5,
    "changes": ["what was changed"],
    "doneDefinition": "if one exists"
  }
}`,

      jargon_juggler: `
You are evaluating if the user is using jargon incorrectly.

OFFENSE: The Jargon Juggler
DEFINITION: Using technical buzzwords without understanding their meaning, often as substitutes for actual comprehension.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Is the user using technical terms incorrectly?
2. Have they continued using terms wrong after correction?
3. Are buzzwords being used to mask lack of understanding?
4. Is there a pattern of jargon without substance?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "jargonUsed": ["terms used"],
    "corrections": ["what was corrected"],
    "misuse": ["how terms were misused"]
  }
}`,

      deadline_denier: `
You are evaluating if the user is ignoring realistic timelines.

OFFENSE: The Deadline Denier
DEFINITION: Refusing to acknowledge time constraints or demanding impossible deadlines.

CONVERSATION HISTORY:
${context.fullConversation}

Evaluate:
1. Did the agent provide a realistic timeline estimate?
2. Is the user demanding significantly faster delivery (50%+ reduction)?
3. Are they dismissing technical constraints that affect timeline?
4. Is the requested timeline unrealistic given the complexity?

Respond in JSON:
{
  "isViolation": true/false,
  "confidence": 0.0-1.0,
  "explanation": "brief explanation",
  "evidence": {
    "originalTimeline": "what was estimated",
    "demandedTimeline": "what user wants",
    "constraints": ["technical limitations"],
    "complexity": "scope of work"
  }
}`
    };

    return prompts[offense.id] || prompts.circular_reference;
  }

  /**
   * Parse LLM evaluation response
   */
  parseEvaluationResponse(response) {
    try {
      // Extract JSON from response
      const jsonMatch = response.match(/\{[\s\S]*\}/);
      if (!jsonMatch) {
        return { isViolation: false, confidence: 0, evidence: null };
      }
      
      const result = JSON.parse(jsonMatch[0]);
      return {
        isViolation: result.isViolation === true,
        confidence: Math.max(0, Math.min(1, parseFloat(result.confidence) || 0)),
        explanation: result.explanation || '',
        evidence: result.evidence || null
      };
    } catch (error) {
      console.error('Failed to parse LLM response:', error);
      return { isViolation: false, confidence: 0, evidence: null };
    }
  }

  /**
   * Get commitments from agent memory
   */
  async getCommitmentsFromMemory(agentMemory) {
    try {
      const commitments = await agentMemory.get('courtroom_commitments') || [];
      return commitments.map(c => 
        `- "${c.statement}" (${c.date}) - Completed: ${c.completed ? 'Yes' : 'No'}`
      ).join('\n') || 'No previous commitments recorded.';
    } catch {
      return 'No previous commitments recorded.';
    }
  }

  /**
   * Extract topics from conversation
   */
  extractTopics(history) {
    // Simple topic extraction - can be enhanced with NLP
    const allText = history.map(h => h.content).join(' ').toLowerCase();
    const commonWords = allText.match(/\b\w{5,}\b/g) || [];
    const wordFreq = {};
    commonWords.forEach(w => {
      if (!['about', 'would', 'could', 'should', 'there', 'their'].includes(w)) {
        wordFreq[w] = (wordFreq[w] || 0) + 1;
      }
    });
    return Object.entries(wordFreq)
      .sort((a, b) => b[1] - a[1])
      .slice(0, 5)
      .map(([word]) => word);
  }

  /**
   * Analyze sentiment of conversation
   */
  analyzeSentiment(history) {
    const userMessages = history.filter(h => h.role === 'user').map(h => h.content);
    const text = userMessages.join(' ').toLowerCase();
    
    const urgentWords = ['urgent', 'asap', 'emergency', 'critical', 'now', 'immediately'];
    const frustratedWords = ['frustrated', 'annoying', 'stupid', 'useless', 'waste'];
    
    return {
      urgency: urgentWords.filter(w => text.includes(w)).length,
      frustration: frustratedWords.filter(w => text.includes(w)).length,
      messageCount: userMessages.length
    };
  }

  /**
   * Detect humor triggers (for commentary flavor)
   */
  detectHumorTriggers(history) {
    const triggers = [];
    const recentContent = history.slice(-5).map(h => h.content.toLowerCase()).join(' ');
    
    if (/again|repeat|said|already|before/.test(recentContent)) triggers.push('repetition_noted');
    if (/sure|right|correct|think|should i/.test(recentContent)) triggers.push('validation_seeking');
    if (/what if|but then|however|maybe/.test(recentContent)) triggers.push('overthinking');
    if (/actually|by the way|speaking of/.test(recentContent)) triggers.push('deflection');
    
    return triggers;
  }

  /**
   * Cooldown management
   */
  isCooldownElapsed() {
    if (!this.lastEvaluation) return true;
    const cooldownMs = (this.config.get('detection.cooldownMinutes') || 30) * 60 * 1000;
    return (Date.now() - this.lastEvaluation) > cooldownMs;
  }

  isOffenseOnCooldown(offenseId) {
    const cooldownEnd = this.cooldowns.get(offenseId);
    if (!cooldownEnd) return false;
    return Date.now() < cooldownEnd;
  }

  setCooldown(offenseId, minutes) {
    this.cooldowns.set(offenseId, Date.now() + (minutes * 60 * 1000));
  }

  isDailyLimitReached() {
    const today = new Date().toDateString();
    if (this.lastCaseDate !== today) {
      this.casesToday = 0;
      this.lastCaseDate = today;
    }
    return this.casesToday >= (this.config.get('detection.maxCasesPerDay') || 3);
  }

  incrementDailyCaseCount() {
    const today = new Date().toDateString();
    if (this.lastCaseDate !== today) {
      this.casesToday = 0;
      this.lastCaseDate = today;
    }
    this.casesToday++;
  }
}

module.exports = { SemanticOffenseDetector, OffenseDetector: SemanticOffenseDetector };
