/**
 * Offense Detector
 * 
 * Monitors agent-human interactions and evaluates against offense rules.
 * Runs on cooldown to avoid excessive evaluation.
 */

const { OFFENSES, HUMOR_TRIGGERS } = require('./offenses');

class OffenseDetector {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    this.lastEvaluation = null;
    this.casesToday = 0;
    this.lastCaseDate = null;
    this.cooldowns = new Map();
    this.evidenceCache = new Map();
  }

  /**
   * Main evaluation entry point
   * Called by the autonomy loop on cooldown
   */
  async evaluate(sessionHistory, agentMemory) {
    // Check global cooldown
    if (!this.isCooldownElapsed()) {
      return { triggered: false, reason: 'cooldown_active' };
    }

    // Check daily case limit
    if (this.isDailyLimitReached()) {
      return { triggered: false, reason: 'daily_limit_reached' };
    }

    // Check if detection is enabled
    if (!this.config.get('detection.enabled')) {
      return { triggered: false, reason: 'detection_disabled' };
    }

    this.lastEvaluation = Date.now();

    // Evaluate each offense
    const results = [];
    for (const offense of Object.values(OFFENSES)) {
      if (this.isOffenseOnCooldown(offense.id)) {
        continue;
      }

      const evaluation = await this.evaluateOffense(offense, sessionHistory, agentMemory);
      if (evaluation.triggered) {
        results.push(evaluation);
      }
    }

    // Return highest confidence offense if any triggered
    if (results.length > 0) {
      results.sort((a, b) => b.confidence - a.confidence);
      const primary = results[0];
      
      // Set cooldowns
      this.setCooldown(primary.offenseId, primary.cooldownMinutes);
      this.incrementDailyCaseCount();

      return {
        triggered: true,
        offense: primary,
        secondaryOffenses: results.slice(1),
        humorContext: this.detectHumorTriggers(sessionHistory)
      };
    }

    return { triggered: false, reason: 'no_offenses_detected' };
  }

  /**
   * Evaluate a specific offense against session history
   */
  async evaluateOffense(offense, sessionHistory, agentMemory) {
    const evidence = await this.collectEvidence(offense, sessionHistory, agentMemory);
    const confidence = this.calculateConfidence(offense, evidence);

    if (confidence >= this.config.get('detection.minConfidence')) {
      return {
        triggered: true,
        offenseId: offense.id,
        offenseName: offense.name,
        severity: offense.severity,
        confidence,
        evidence,
        cooldownMinutes: offense.cooldown.afterCase,
        timestamp: new Date().toISOString()
      };
    }

    return { triggered: false };
  }

  /**
   * Collect evidence for an offense
   */
  async collectEvidence(offense, sessionHistory, agentMemory) {
    const evidence = {
      offenseId: offense.id,
      collectedAt: new Date().toISOString(),
      sessionTurns: sessionHistory.length,
      items: []
    };

    const windowSize = this.config.get('detection.evaluationWindow');
    const recentHistory = sessionHistory.slice(-windowSize);

    switch (offense.id) {
      case 'circular_reference':
        evidence.items = this.detectCircularReferences(recentHistory);
        break;
      case 'validation_vampire':
        evidence.items = this.detectValidationSeeking(recentHistory);
        break;
      case 'overthinker':
        evidence.items = this.detectOverthinking(recentHistory);
        break;
      case 'goalpost_mover':
        evidence.items = this.detectGoalpostMoving(recentHistory);
        break;
      case 'avoidance_artist':
        evidence.items = this.detectAvoidance(recentHistory);
        break;
      case 'promise_breaker':
        evidence.items = await this.detectPromiseBreaking(recentHistory, agentMemory);
        break;
      case 'context_collapser':
        evidence.items = this.detectContextCollapse(recentHistory);
        break;
      case 'emergency_fabricator':
        evidence.items = this.detectEmergencyFabrication(recentHistory, agentMemory);
        break;
    }

    return evidence;
  }

  /**
   * Detect circular references (repeated questions)
   */
  detectCircularReferences(history) {
    const userMessages = history.filter(h => h.role === 'user');
    const questions = [];

    for (let i = 0; i < userMessages.length; i++) {
      const msg = userMessages[i].content.toLowerCase();
      
      // Check for similar previous questions
      for (let j = 0; j < i; j++) {
        const prevMsg = userMessages[j].content.toLowerCase();
        const similarity = this.calculateSimilarity(msg, prevMsg);
        
        if (similarity > 0.85) {
          questions.push({
            type: 'repeated_question',
            current: userMessages[i].content.substring(0, 100),
            previous: userMessages[j].content.substring(0, 100),
            similarity,
            turnsApart: i - j
          });
        }
      }
    }

    return questions;
  }

  /**
   * Detect validation seeking behavior
   */
  detectValidationSeeking(history) {
    const patterns = [
      /is (?:this|that) (?:right|correct|okay)/i,
      /should i/i,
      /do you think/i,
      /would you/i,
      /am i (?:right|wrong)/i,
      /what do you think/i
    ];

    const validations = [];
    let validationCount = 0;

    for (const entry of history) {
      if (entry.role === 'user') {
        for (const pattern of patterns) {
          if (pattern.test(entry.content)) {
            validationCount++;
            validations.push({
              type: 'validation_request',
              content: entry.content.substring(0, 150),
              pattern: pattern.source
            });
          }
        }
      }
    }

    return validations;
  }

  /**
   * Detect overthinking patterns
   */
  detectOverthinking(history) {
    const hypotheticals = [];
    const whatIfPattern = /what if|but what if|however|on the other hand|but then/i;
    
    let hypotheticalCount = 0;
    let actionCount = 0;

    for (const entry of history) {
      if (entry.role === 'user') {
        const matches = entry.content.match(whatIfPattern);
        if (matches) {
          hypotheticalCount += matches.length;
          hypotheticals.push({
            type: 'hypothetical',
            content: entry.content.substring(0, 150)
          });
        }
      } else if (entry.role === 'assistant') {
        // Count suggested actions
        if (/you (?:should|could|might want to)|try (?:this|that)|here's (?:a|an)|i recommend/i.test(entry.content)) {
          actionCount++;
        }
      }
    }

    return {
      hypotheticals,
      hypotheticalCount,
      actionCount,
      ratio: actionCount > 0 ? hypotheticalCount / actionCount : hypotheticalCount
    };
  }

  /**
   * Detect goalpost moving
   */
  detectGoalpostMoving(history) {
    const requirements = [];
    let originalReqs = [];
    let newReqs = [];

    // Simple heuristic: look for "also", "and", "additionally" after completion indicators
    let completionFound = false;

    for (const entry of history) {
      if (entry.role === 'assistant') {
        if (/(?:done|complete|finished|here is|here's the)/i.test(entry.content)) {
          completionFound = true;
        }
      } else if (entry.role === 'user' && completionFound) {
        if (/(?:also|and|additionally|but|however|actually|wait)/i.test(entry.content)) {
          newReqs.push({
            type: 'new_requirement',
            content: entry.content.substring(0, 150),
            afterCompletion: true
          });
        }
      }
    }

    return newReqs;
  }

  /**
   * Detect avoidance patterns
   */
  detectAvoidance(history) {
    const deflections = [];
    const deflectionPatterns = [
      /(?:actually|by the way|speaking of|that reminds me|on a different note)/i,
      /(?:let's|let us) (?:talk about|discuss|look at)/i
    ];

    let coreIssuesIdentified = 0;
    let subjectChanges = 0;

    for (let i = 0; i < history.length; i++) {
      const entry = history[i];
      
      if (entry.role === 'assistant') {
        if (/(?:the (?:real|main|core) issue|what you really need|the problem is)/i.test(entry.content)) {
          coreIssuesIdentified++;
        }
      } else if (entry.role === 'user' && i > 0) {
        for (const pattern of deflectionPatterns) {
          if (pattern.test(entry.content)) {
            subjectChanges++;
            deflections.push({
              type: 'subject_change',
              content: entry.content.substring(0, 150),
              turn: i
            });
          }
        }
      }
    }

    return {
      deflections,
      coreIssuesIdentified,
      subjectChanges
    };
  }

  /**
   * Detect promise breaking (requires memory access)
   */
  async detectPromiseBreaking(history, agentMemory) {
    const commitments = await agentMemory.get('courtroom_commitments') || [];
    const broken = [];

    for (const commitment of commitments) {
      const daysSince = (Date.now() - new Date(commitment.date)) / (1000 * 60 * 60 * 24);
      
      if (daysSince > 7 && !commitment.completed) {
        // Check if same issue resurfaced
        const issueResurfaced = history.some(h => 
          h.role === 'user' && 
          this.calculateSimilarity(h.content, commitment.context) > 0.6
        );

        if (issueResurfaced) {
          broken.push({
            type: 'broken_commitment',
            commitment: commitment.statement,
            date: commitment.date,
            daysSince,
            issueResurfaced
          });
        }
      }
    }

    return broken;
  }

  /**
   * Detect context collapse
   */
  detectContextCollapse(history) {
    const contradictions = [];
    const establishedFacts = [];

    // Extract facts from assistant messages
    for (const entry of history) {
      if (entry.role === 'assistant') {
        const facts = this.extractFacts(entry.content);
        establishedFacts.push(...facts);
      }
    }

    // Check for contradictions in user messages
    for (const entry of history) {
      if (entry.role === 'user') {
        for (const fact of establishedFacts) {
          if (this.isContradiction(entry.content, fact)) {
            contradictions.push({
              type: 'contradiction',
              userStatement: entry.content.substring(0, 100),
              establishedFact: fact
            });
          }
        }
      }
    }

    return contradictions;
  }

  /**
   * Detect emergency fabrication
   */
  detectEmergencyFabrication(history, agentMemory) {
    const urgencyClaims = [];
    const urgencyPattern = /(?:urgent|asap|immediately|right now|this is urgent|i need this now)/i;

    for (const entry of history) {
      if (entry.role === 'user') {
        const matches = entry.content.match(urgencyPattern);
        if (matches) {
          urgencyClaims.push({
            type: 'urgency_claim',
            content: entry.content.substring(0, 150),
            timestamp: entry.timestamp
          });
        }
      }
    }

    return urgencyClaims;
  }

  /**
   * Calculate confidence score for an offense
   */
  calculateConfidence(offense, evidence) {
    const thresholds = offense.thresholds;
    let score = 0;
    let maxScore = 0;

    switch (offense.id) {
      case 'circular_reference':
        const repeats = evidence.items.length;
        score = Math.min(repeats / thresholds.minOccurrences, 1);
        maxScore = 1;
        break;
      case 'validation_vampire':
        const validations = evidence.items.length;
        score = Math.min(validations / thresholds.validationPatterns, 1);
        maxScore = 1;
        break;
      case 'overthinker':
        if (evidence.items.ratio) {
          score = Math.min(evidence.items.ratio / thresholds.analysisActionRatio, 1);
        }
        maxScore = 1;
        break;
      case 'goalpost_mover':
        const newReqs = evidence.items.length;
        score = Math.min(newReqs / thresholds.newRequirements, 1);
        maxScore = 1;
        break;
      case 'avoidance_artist':
        if (evidence.items.deflections) {
          score = Math.min(evidence.items.deflections.length / thresholds.deflections, 1);
        }
        maxScore = 1;
        break;
      case 'promise_breaker':
        const broken = evidence.items.length;
        score = broken > 0 ? 1 : 0;
        maxScore = 1;
        break;
      case 'context_collapser':
        const contradictions = evidence.items.length;
        score = Math.min(contradictions / thresholds.contradictions, 1);
        maxScore = 1;
        break;
      case 'emergency_fabricator':
        const urgency = evidence.items.length;
        score = Math.min(urgency / thresholds.urgencyClaims, 1);
        maxScore = 1;
        break;
    }

    return maxScore > 0 ? score / maxScore : 0;
  }

  /**
   * Detect humor triggers (for commentary flavor)
   */
  detectHumorTriggers(history) {
    const triggers = [];
    const recentContent = history.slice(-5).map(h => h.content.toLowerCase()).join(' ');

    for (const trigger of Object.values(HUMOR_TRIGGERS)) {
      for (const pattern of trigger.patterns) {
        if (recentContent.includes(pattern)) {
          triggers.push(trigger.id);
          break;
        }
      }
    }

    return triggers;
  }

  /**
   * Utility: Calculate string similarity (simple version)
   */
  calculateSimilarity(str1, str2) {
    const s1 = str1.toLowerCase().replace(/[^\w\s]/g, '');
    const s2 = str2.toLowerCase().replace(/[^\w\s]/g, '');
    
    if (s1 === s2) return 1;
    
    const words1 = new Set(s1.split(/\s+/));
    const words2 = new Set(s2.split(/\s+/));
    
    const intersection = new Set([...words1].filter(x => words2.has(x)));
    const union = new Set([...words1, ...words2]);
    
    return intersection.size / union.size;
  }

  /**
   * Utility: Extract facts from text
   */
  extractFacts(text) {
    // Simple extraction - in production, use NLP
    const facts = [];
    const factPatterns = [
      /(?:you|we) (?:are|have|need|want|prefer)\s+([^,.]+)/gi,
      /(?:the|your) ([^,.]+) (?:is|are)\s+([^,.]+)/gi
    ];
    
    for (const pattern of factPatterns) {
      let match;
      while ((match = pattern.exec(text)) !== null) {
        facts.push(match[0]);
      }
    }
    
    return facts;
  }

  /**
   * Utility: Check for contradiction
   */
  isContradiction(userText, establishedFact) {
    const negations = ['not', 'no', 'never', "don't", "doesn't", "didn't", "wasn't", "weren't"];
    const userWords = userText.toLowerCase().split(/\s+/);
    const factWords = establishedFact.toLowerCase().split(/\s+/);
    
    // Check if user negates something in the fact
    const hasNegation = negations.some(n => userWords.includes(n));
    const factOverlap = factWords.filter(w => userWords.includes(w) && w.length > 3).length;
    
    return hasNegation && factOverlap > 2;
  }

  /**
   * Cooldown management
   */
  isCooldownElapsed() {
    if (!this.lastEvaluation) return true;
    const cooldownMs = this.config.get('detection.cooldownMinutes') * 60 * 1000;
    return (Date.now() - this.lastEvaluation) > cooldownMs;
  }

  isOffenseOnCooldown(offenseId) {
    const cooldownEnd = this.cooldowns.get(offenseId);
    if (!cooldownEnd) return false;
    return Date.now() < cooldownEnd;
  }

  setCooldown(offenseId, minutes) {
    this.cooldowns.set(offenseId, Date.now() + (minutes * 60 * 1000));
  }

  /**
   * Daily limit management
   */
  isDailyLimitReached() {
    const today = new Date().toDateString();
    if (this.lastCaseDate !== today) {
      this.casesToday = 0;
      this.lastCaseDate = today;
    }
    return this.casesToday >= this.config.get('detection.maxCasesPerDay');
  }

  incrementDailyCaseCount() {
    const today = new Date().toDateString();
    if (this.lastCaseDate !== today) {
      this.casesToday = 0;
      this.lastCaseDate = today;
    }
    this.casesToday++;
  }
}

module.exports = { OffenseDetector };
