/**
 * Debug Logger for ClawTrial
 * Logs all courtroom activity for troubleshooting
 */

const fs = require('fs');
const path = require('path');

// Safely get config dir with fallback
function getSafeConfigDir() {
  try {
    const { getConfigDir } = require('./environment');
    return getConfigDir();
  } catch (e) {
    // Fallback to .clawdbot if environment module fails
    return path.join(process.env.HOME || '', '.clawdbot');
  }
}

class DebugLogger {
  constructor() {
    this.logs = [];
    this.maxLogs = 1000;
    this.logFile = path.join(getSafeConfigDir(), 'courtroom_debug.log');
    this.enabled = true;
  }

  log(level, component, message, data = null) {
    if (!this.enabled) return;

    const entry = {
      timestamp: new Date().toISOString(),
      level,
      component,
      message,
      data
    };

    this.logs.push(entry);

    // Keep only last maxLogs entries
    if (this.logs.length > this.maxLogs) {
      this.logs.shift();
    }

    // Also write to file
    this.writeToFile(entry);

    // Console output for debugging
    if (process.env.COURTROOM_DEBUG === 'true') {
      console.log(`[${level}] ${component}: ${message}`);
    }
  }

  writeToFile(entry) {
    try {
      const line = JSON.stringify(entry) + '\n';
      fs.appendFileSync(this.logFile, line);
    } catch (err) {
      // Silent fail - don't break functionality for logging
    }
  }

  info(component, message, data) {
    this.log('INFO', component, message, data);
  }

  warn(component, message, data) {
    this.log('WARN', component, message, data);
  }

  error(component, message, data) {
    this.log('ERROR', component, message, data);
  }

  debug(component, message, data) {
    this.log('DEBUG', component, message, data);
  }

  getLogs(level = null, component = null, limit = 100) {
    let filtered = this.logs;

    if (level) {
      filtered = filtered.filter(l => l.level === level);
    }

    if (component) {
      filtered = filtered.filter(l => l.component === component);
    }

    return filtered.slice(-limit);
  }

  getRecentLogs(minutes = 30) {
    const cutoff = Date.now() - (minutes * 60 * 1000);
    return this.logs.filter(l => new Date(l.timestamp).getTime() > cutoff);
  }

  clearLogs() {
    this.logs = [];
    try {
      fs.unlinkSync(this.logFile);
    } catch (err) {
      // File might not exist
    }
  }

  printStatus() {
    const recent = this.getRecentLogs(60);
    const errors = recent.filter(l => l.level === 'ERROR');
    const warnings = recent.filter(l => l.level === 'WARN');

    console.log('\n🏛️  ClawTrial Debug Status\n');
    console.log('===========================\n');
    console.log(`Total logs in memory: ${this.logs.length}`);
    console.log(`Logs in last hour: ${recent.length}`);
    console.log(`Errors: ${errors.length}`);
    console.log(`Warnings: ${warnings.length}`);
    console.log(`Log file: ${this.logFile}`);
    console.log(`Debug mode: ${process.env.COURTROOM_DEBUG === 'true' ? 'ON' : 'OFF'}`);
    console.log('\nRecent activity:');
    
    const last10 = this.logs.slice(-10);
    last10.forEach(log => {
      console.log(`  [${log.level}] ${log.component}: ${log.message.substring(0, 60)}`);
    });
  }

  printFullLog(limit = 50) {
    console.log('\n🏛️  ClawTrial Full Debug Log\n');
    console.log('=============================\n');
    
    const logs = this.logs.slice(-limit);
    logs.forEach(log => {
      console.log(`\n[${log.timestamp}] ${log.level} - ${log.component}`);
      console.log(`  ${log.message}`);
      if (log.data) {
        console.log(`  Data:`, JSON.stringify(log.data, null, 2).substring(0, 200));
      }
    });
  }
}

// Singleton instance
const logger = new DebugLogger();

module.exports = { DebugLogger, logger };
