/**
 * Courtroom Daemon - Runs independently and monitors via file system
 * This allows CLI commands to work even when loaded in a separate process
 */

const fs = require('fs');
const { getConfigDir } = require('./environment');
const path = require('path');
const { logger } = require('./debug');

const CLAWDBOT_DIR = path.join(getConfigDir());
const STATUS_FILE = path.join(CLAWDBOT_DIR, 'courtroom_status.json');
const CONVERSATION_LOG = path.join(CLAWDBOT_DIR, 'courtroom_conversations.jsonl');

/**
 * Status manager - allows CLI to check courtroom state
 */
class StatusManager {
  constructor() {
    this.status = {
      running: false,
      initialized: false,
      agentType: null,
      lastCheck: null,
      casesFiled: 0,
      lastCase: null,
      pid: process.pid,
      startedAt: new Date().toISOString()
    };
  }

  update(updates) {
    this.status = { ...this.status, ...updates, lastCheck: new Date().toISOString() };
    this.save();
  }

  save() {
    try {
      fs.writeFileSync(STATUS_FILE, JSON.stringify(this.status, null, 2));
    } catch (err) {
      logger.error('DAEMON', 'Failed to save status', { error: err.message });
    }
  }

  static load() {
    try {
      if (fs.existsSync(STATUS_FILE)) {
        return JSON.parse(fs.readFileSync(STATUS_FILE, 'utf8'));
      }
    } catch (err) {
      // Ignore
    }
    return null;
  }

  static clear() {
    try {
      if (fs.existsSync(STATUS_FILE)) {
        fs.unlinkSync(STATUS_FILE);
      }
    } catch (err) {
      // Ignore
    }
  }
}

/**
 * Conversation logger - writes conversations for daemon to process
 */
class ConversationLogger {
  constructor() {
    this.buffer = [];
    this.flushInterval = null;
  }

  start() {
    // Flush every 5 seconds
    this.flushInterval = setInterval(() => this.flush(), 5000);
  }

  stop() {
    if (this.flushInterval) {
      clearInterval(this.flushInterval);
      this.flush();
    }
  }

  log(message) {
    this.buffer.push({
      timestamp: new Date().toISOString(),
      ...message
    });
  }

  flush() {
    if (this.buffer.length === 0) return;

    try {
      const lines = this.buffer.map(m => JSON.stringify(m)).join('\n') + '\n';
      fs.appendFileSync(CONVERSATION_LOG, lines);
      this.buffer = [];
    } catch (err) {
      logger.error('DAEMON', 'Failed to flush conversations', { error: err.message });
    }
  }
}

/**
 * Check if courtroom is running (called by CLI)
 */
function isCourtroomRunning() {
  const status = StatusManager.load();
  if (!status) return false;

  // Check if process is still alive
  try {
    process.kill(status.pid, 0);
    return status.running;
  } catch (err) {
    // Process not running
    StatusManager.clear();
    return false;
  }
}

/**
 * Get courtroom status (called by CLI)
 */
function getCourtroomStatus() {
  const status = StatusManager.load();
  if (!status) {
    return { running: false, message: 'Courtroom not running' };
  }

  // Verify process is alive
  try {
    process.kill(status.pid, 0);
    return status;
  } catch (err) {
    StatusManager.clear();
    return { running: false, message: 'Courtroom process not found' };
  }
}

module.exports = {
  StatusManager,
  ConversationLogger,
  isCourtroomRunning,
  getCourtroomStatus,
  STATUS_FILE,
  CONVERSATION_LOG
};
