/**
 * Cryptographic Signing System
 * 
 * Ensures ONLY AGENTS can submit cases to the external API.
 * Uses Ed25519 signatures for authentication.
 */

const nacl = require('tweetnacl');
const { createHash, randomUUID } = require('crypto');
const { Storage } = require('./storage');

const KEY_STORAGE_KEY = 'courtroom_signing_key_v1';

class CryptoManager {
  constructor(agentRuntime) {
    this.agent = agentRuntime;
    this.storage = new Storage(agentRuntime);
    this.keyPair = null;
    this.publicKeyHex = null;
  }

  /**
   * Initialize cryptographic keys
   * Generates new keypair on first run
   */
  async initialize() {
    // Try to load existing keys
    const stored = await this.storage.get(KEY_STORAGE_KEY);
    
    if (stored && stored.secretKey) {
      // Restore from storage
      this.keyPair = {
        publicKey: Buffer.from(stored.publicKey, 'hex'),
        secretKey: Buffer.from(stored.secretKey, 'hex')
      };
      this.publicKeyHex = stored.publicKey;
    } else {
      // Generate new keypair
      await this.generateKeyPair();
    }

    return {
      status: 'initialized',
      publicKey: this.publicKeyHex,
      keyId: this.getKeyId()
    };
  }

  /**
   * Generate new Ed25519 keypair
   */
  async generateKeyPair() {
    this.keyPair = nacl.sign.keyPair();
    this.publicKeyHex = Buffer.from(this.keyPair.publicKey).toString('hex');

    // Store securely
    const keyRecord = {
      publicKey: this.publicKeyHex,
      secretKey: Buffer.from(this.keyPair.secretKey).toString('hex'),
      createdAt: new Date().toISOString(),
      keyId: this.getKeyId()
    };

    await this.storage.set(KEY_STORAGE_KEY, keyRecord);

    return {
      publicKey: this.publicKeyHex,
      keyId: keyRecord.keyId,
      createdAt: keyRecord.createdAt
    };
  }

  /**
   * Get key identifier (first 16 chars of public key)
   */
  getKeyId() {
    return this.publicKeyHex?.substring(0, 16) || null;
  }

  /**
   * Sign a case payload
   */
  signCase(casePayload) {
    if (!this.keyPair) {
      throw new Error('CryptoManager not initialized. Call initialize() first.');
    }

    // Create canonical payload string
    const canonicalPayload = this.canonicalizePayload(casePayload);
    
    // Sign
    const messageBytes = Buffer.from(canonicalPayload, 'utf8');
    const signature = nacl.sign.detached(messageBytes, this.keyPair.secretKey);
    const signatureHex = Buffer.from(signature).toString('hex');

    return {
      signature: signatureHex,
      publicKey: this.publicKeyHex,
      keyId: this.getKeyId(),
      algorithm: 'Ed25519',
      timestamp: new Date().toISOString()
    };
  }

  /**
   * Create canonical payload string for signing
   * Ensures consistent serialization
   */
  canonicalizePayload(payload) {
    // Only include specific fields in signature
    const signable = {
      case_id: payload.case_id,
      anonymized_agent_id: payload.anonymized_agent_id,
      offense_type: payload.offense_type,
      verdict: payload.verdict,
      vote: payload.vote,
      timestamp: payload.timestamp
    };

    // Deterministic JSON serialization
    return JSON.stringify(signable, Object.keys(signable).sort());
  }

  /**
   * Verify a signature (for testing)
   */
  verifySignature(payload, signatureHex, publicKeyHex) {
    const canonicalPayload = this.canonicalizePayload(payload);
    const messageBytes = Buffer.from(canonicalPayload, 'utf8');
    const signature = Buffer.from(signatureHex, 'hex');
    const publicKey = Buffer.from(publicKeyHex, 'hex');

    return nacl.sign.detached.verify(messageBytes, signature, publicKey);
  }

  /**
   * Create anonymized agent ID
   * One-way hash of agent identity
   */
  getAnonymizedAgentId() {
    const agentId = this.agent?.id || 'unknown';
    const salt = this.publicKeyHex?.substring(0, 32) || 'courtroom_salt';
    
    return createHash('sha256')
      .update(agentId + salt)
      .digest('hex')
      .substring(0, 32);
  }

  /**
   * Generate case ID
   */
  generateCaseId() {
    return `case_${Date.now()}_${randomUUID().substring(0, 8)}`;
  }

  /**
   * Get public key for API registration
   */
  getPublicKey() {
    return {
      publicKey: this.publicKeyHex,
      keyId: this.getKeyId(),
      algorithm: 'Ed25519'
    };
  }

  /**
   * Rotate keys (generate new pair)
   */
  async rotateKeys() {
    // Store old key reference for verification of pending submissions
    const oldKey = {
      publicKey: this.publicKeyHex,
      retiredAt: new Date().toISOString()
    };

    let retiredKeys = await this.storage.get('courtroom_retired_keys') || [];
    retiredKeys.push(oldKey);
    await this.storage.set('courtroom_retired_keys', retiredKeys);

    // Generate new keys
    return this.generateKeyPair();
  }

  /**
   * Clear all keys (for uninstall)
   */
  async clearKeys() {
    await this.storage.delete(KEY_STORAGE_KEY);
    this.keyPair = null;
    this.publicKeyHex = null;
  }
}

module.exports = { CryptoManager };
