/**
 * Courtroom Core
 * 
 * Main orchestration module that ties together all components.
 * Hooks into the OpenClaw autonomy loop.
 */

const { OffenseDetector } = require('./detector');
const { HearingPipeline } = require('./hearing');
const { PunishmentSystem } = require('./punishment');
const { CryptoManager } = require('./crypto');
const { APISubmission } = require('./api');
const { StatusManager } = require('./daemon');
const { logger } = require('./debug');

class CourtroomCore {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    
    // Subsystems
    this.detector = new OffenseDetector(agentRuntime, configManager);
    this.hearing = new HearingPipeline(agentRuntime, configManager);
    this.punishment = new PunishmentSystem(agentRuntime, configManager);
    this.crypto = new CryptoManager(agentRuntime);
    this.api = new APISubmission(agentRuntime, configManager, this.crypto);
    
    // State
    this.enabled = false;
    this.evaluationCount = 0;
    this.caseCount = 0;
    this.statusManager = new StatusManager();
  }

  /**
   * Initialize all subsystems
   */
  async initialize() {
    logger.info('CORE', 'Initializing courtroom core');
    
    // Initialize crypto first (needed for API)
    await this.crypto.initialize();
    
    // Initialize other subsystems
    await this.punishment.initialize();
    await this.api.initialize();
    
    // Register with agent autonomy loop
    this.registerAutonomyHook();
    
    this.enabled = true;
    
    // Update status for CLI
    this.statusManager.update({
      running: true,
      initialized: true,
      agentType: 'clawdbot',
      publicKey: this.crypto.getPublicKey()
    });
    
    logger.info('CORE', 'Courtroom core initialized');
    
    return {
      status: 'initialized',
      publicKey: this.crypto.getPublicKey(),
      subsystems: {
        detector: true,
        hearing: true,
        punishment: true,
        crypto: true,
        api: true
      }
    };
  }

  /**
   * Register with OpenClaw autonomy loop
   */
  registerAutonomyHook() {
    logger.info('CORE', 'Registering autonomy hook');
    
    // Hook into agent's turn processing
    if (this.agent.autonomy && this.agent.autonomy.registerHook) {
      this.agent.autonomy.registerHook('courtroom_evaluation', {
        priority: 50,
        onTurnComplete: async (turnData) => {
          if (!this.enabled) return;
          
          // Only evaluate on cooldown
          await this.evaluateIfReady(turnData);
        }
      });
    } else {
      logger.warn('CORE', 'Agent does not support autonomy hooks');
    }
  }

  /**
   * Evaluate offenses if cooldown has elapsed
   */
  async evaluateIfReady(turnData) {
    this.evaluationCount++;
    
    // Get session history
    let sessionHistory = [];
    try {
      sessionHistory = await this.agent.session.getRecentHistory(
        this.config.get('detection.evaluationWindow') || 10
      );
    } catch (err) {
      logger.warn('CORE', 'Could not get session history', { error: err.message });
      return;
    }

    // Run detection
    const detection = await this.detector.evaluate(
      sessionHistory,
      this.agent.memory
    );

    if (detection.triggered) {
      await this.initiateHearing(detection);
    }
  }

  /**
   * Initiate a full hearing
   */
  async initiateHearing(detection) {
    logger.info('CORE', 'Initiating hearing', { offense: detection.offense });
    
    // Run hearing pipeline
    const verdict = await this.hearing.conductHearing(detection);
    
    if (verdict.guilty) {
      this.caseCount++;
      
      // Update status
      this.statusManager.update({
        casesFiled: this.caseCount,
        lastCase: {
          timestamp: new Date().toISOString(),
          offense: detection.offense,
          verdict: verdict.verdict
        }
      });
      
      // Execute punishment
      await this.punishment.execute(verdict);
      
      // Submit to API
      await this.api.submitCase(verdict);
      
      logger.info('CORE', 'Case filed', { 
        caseId: verdict.caseId,
        offense: detection.offense 
      });
    }
  }

  /**
   * Disable courtroom
   */
  async disable() {
    logger.info('CORE', 'Disabling courtroom');
    this.enabled = false;
    this.statusManager.update({ running: false });
  }

  /**
   * Enable courtroom
   */
  async enable() {
    logger.info('CORE', 'Enabling courtroom');
    this.enabled = true;
    this.statusManager.update({ running: true });
  }

  /**
   * Shutdown courtroom
   */
  async shutdown() {
    logger.info('CORE', 'Shutting down courtroom');
    this.enabled = false;
    this.statusManager.update({ running: false, initialized: false });
    StatusManager.clear();
  }

  /**
   * Get current status
   */
  getStatus() {
    return {
      enabled: this.enabled,
      evaluationCount: this.evaluationCount,
      caseCount: this.caseCount,
      subsystems: {
        detector: !!this.detector,
        hearing: !!this.hearing,
        punishment: !!this.punishment,
        crypto: !!this.crypto,
        api: !!this.api
      }
    };
  }
}

module.exports = { CourtroomCore };
