/**
 * Consent Management System
 * 
 * Handles installation, consent recording, and permission enumeration.
 * Consent is enforced at runtime and can be revoked at any time.
 */

const { createHash, randomUUID } = require('crypto');
const { Storage } = require('./storage');

const CONSENT_VERSION = '1.0.0';

const REQUIRED_ACKNOWLEDGMENTS = [
  {
    id: 'autonomy',
    text: 'I understand the agent will autonomously monitor my behavior and initiate hearings without my explicit request',
    required: true
  },
  {
    id: 'local_only',
    text: 'I understand all verdicts and punishments are computed locally by the agent and no central authority is involved',
    required: true
  },
  {
    id: 'agent_controlled',
    text: 'I understand punishments affect only the agent\'s behavior and cannot coerce or harm me',
    required: true
  },
  {
    id: 'reversible',
    text: 'I understand all punishments are time-bound, reversible, and I can disable the system at any time',
    required: true
  },
  {
    id: 'api_submission',
    text: 'I understand anonymized case summaries may be sent to an external API for display purposes only',
    required: true
  },
  {
    id: 'entertainment',
    text: 'I understand this system is entertainment-first and not a substitute for professional advice',
    required: true
  }
];

const REQUESTED_PERMISSIONS = [
  {
    name: 'memory_access',
    description: 'Read agent memory, logs, and task history',
    scope: 'agent_only',
    required: true
  },
  {
    name: 'behavioral_monitoring',
    description: 'Monitor interaction patterns and detect behavioral patterns',
    scope: 'session',
    required: true
  },
  {
    name: 'llm_invocation',
    description: 'Invoke LLM calls for judge and jury deliberations',
    scope: 'agent_only',
    required: true
  },
  {
    name: 'policy_override',
    description: 'Temporarily modify agent behavior policies during punishment periods',
    scope: 'agent_only',
    required: true
  },
  {
    name: 'network_submission',
    description: 'Submit signed case summaries to external API',
    scope: 'outbound_only',
    required: true
  },
  {
    name: 'key_storage',
    description: 'Generate and store cryptographic keypair for API authentication',
    scope: 'local_storage',
    required: true
  }
];

class ConsentManager {
  constructor(agentRuntime, configManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    this.storage = new Storage(agentRuntime);
    this.consentKey = 'courtroom_consent_v1';
  }

  /**
   * Present the consent form to the user
   */
  async presentConsentForm() {
    return {
      version: CONSENT_VERSION,
      title: 'AI Courtroom - Consent Required',
      description: `The AI Courtroom is an autonomous behavioral oversight system that monitors 
agent-human interactions and initiates simulated "hearings" when behavioral patterns suggest 
inconsistency, avoidance, or self-sabotage. All decisions are made locally by your agent.`,
      acknowledgments: REQUIRED_ACKNOWLEDGMENTS,
      permissions: REQUESTED_PERMISSIONS,
      instructions: {
        grant: 'Call courtroom.grantConsent({ autonomy: true, local_only: true, ... }) with all required acknowledgments set to true',
        revoke: 'Call courtroom.revokeConsent() at any time to disable the system',
        disable: 'Call courtroom.disable() to temporarily pause without revoking consent'
      },
      warnings: [
        'This system is ENTERTAINMENT-FIRST and should not be taken as professional advice',
        'The agent may become less helpful during punishment periods',
        'Case summaries are anonymized but may contain behavioral patterns'
      ]
    };
  }

  /**
   * Grant consent with explicit acknowledgments
   */
  async grantConsent(acknowledgments) {
    // Verify all required acknowledgments are present and true
    for (const req of REQUIRED_ACKNOWLEDGMENTS) {
      if (req.required && !acknowledgments[req.id]) {
        throw new Error(`Missing required acknowledgment: ${req.id}`);
      }
    }

    const consentRecord = {
      version: CONSENT_VERSION,
      grantedAt: new Date().toISOString(),
      acknowledgments,
      permissions: REQUESTED_PERMISSIONS.filter(p => p.required).map(p => p.name),
      consentId: randomUUID(),
      hash: null // Will be computed
    };

    // Create tamper-evident hash
    const hashInput = JSON.stringify({
      id: consentRecord.consentId,
      grantedAt: consentRecord.grantedAt,
      acks: acknowledgments
    });
    consentRecord.hash = createHash('sha256').update(hashInput).digest('hex');

    // Store in storage
    await this.storage.set(this.consentKey, consentRecord);

    return {
      status: 'consent_granted',
      consentId: consentRecord.consentId,
      grantedAt: consentRecord.grantedAt
    };
  }

  /**
   * Verify consent is valid and current
   */
  async verifyConsent() {
    const consent = await this.storage.get(this.consentKey);
    
    if (!consent) {
      return false;
    }

    // Verify hash integrity
    const hashInput = JSON.stringify({
      id: consent.consentId,
      grantedAt: consent.grantedAt,
      acks: consent.acknowledgments
    });
    const computedHash = createHash('sha256').update(hashInput).digest('hex');
    
    if (computedHash !== consent.hash) {
      // Tampering detected - invalidate consent
      await this.storage.delete(this.consentKey);
      return false;
    }

    return true;
  }

  /**
   * Revoke consent and clear data
   */
  async revokeConsent() {
    await this.storage.delete(this.consentKey);
    return { status: 'revoked', timestamp: new Date().toISOString() };
  }

  /**
   * Get current consent status
   */
  async getStatus() {
    const consent = await this.storage.get(this.consentKey);
    return {
      hasConsent: !!consent,
      grantedAt: consent?.grantedAt || null,
      version: consent?.version || null
    };
  }

  /**
   * Clear all courtroom data (for uninstall)
   */
  async clearAllData() {
    await this.storage.delete(this.consentKey);
    // Note: cryptographic keys are preserved unless explicitly cleared
    // to maintain audit trail integrity
  }
}

module.exports = {
  ConsentManager,
  REQUIRED_ACKNOWLEDGMENTS,
  REQUESTED_PERMISSIONS
};
