/**
 * Auto-start module for ClawDBot
 * Automatically initializes courtroom if consent was granted during install
 */

const fs = require('fs');
const path = require('path');
const { Courtroom } = require('./index');
const { logger } = require('./debug');
const { StatusManager } = require('./daemon');

const CLAWDBOT_DIR = path.join(getConfigDir());
const CONFIG_PATH = path.join(CLAWDBOT_DIR, 'courtroom_config.json');

// Auto-detect ClawDBot environment
function isClawDBot() {
  const checks = {
    env: process.env.CLAUDBOT_ENV === 'true',
    globalAgent: typeof global.clawdbotAgent !== 'undefined',
    globalAgentAlt: typeof global.agent !== 'undefined',
    configDir: fs.existsSync('/home/angad/.clawdbot'),
    configDirAlt: fs.existsSync(CLAWDBOT_DIR)
  };

  logger.debug('AUTOSTART', 'Environment checks', checks);

  return checks.env || checks.globalAgent || checks.globalAgentAlt || checks.configDir || checks.configDirAlt;
}

// Get agent runtime from various possible locations
function getAgentRuntime() {
  const sources = [
    { name: 'global.clawdbotAgent', agent: global.clawdbotAgent },
    { name: 'global.agent', agent: global.agent },
    { name: 'process.clawdbotAgent', agent: process.clawdbotAgent }
  ];

  for (const source of sources) {
    if (source.agent) {
      logger.info('AUTOSTART', `Found agent at ${source.name}`);
      return source.agent;
    }
  }

  logger.warn('AUTOSTART', 'No agent runtime found in global scope');
  return null;
}

// Check if config exists and has consent
function checkConfig() {
  if (!fs.existsSync(CONFIG_PATH)) {
    logger.info('AUTOSTART', 'No config found, skipping auto-start');
    return { exists: false, config: null };
  }

  try {
    const config = JSON.parse(fs.readFileSync(CONFIG_PATH, 'utf8'));
    logger.info('AUTOSTART', 'Config loaded', { 
      hasConsent: config.consent?.granted,
      enabled: config.enabled !== false 
    });
    return { exists: true, config };
  } catch (err) {
    logger.error('AUTOSTART', 'Failed to parse config', { error: err.message });
    return { exists: false, config: null };
  }
}

// Auto-initialize if in ClawDBot and consent granted
async function autoStart() {
  logger.info('AUTOSTART', 'Starting auto-start sequence');

  if (!isClawDBot()) {
    logger.info('AUTOSTART', 'Not in ClawDBot environment, skipping');
    return null;
  }

  const { exists, config } = checkConfig();
  
  if (!exists) {
    logger.info('AUTOSTART', 'No config, user needs to run setup');
    console.log('\n🏛️  ClawTrial not configured. Run: clawtrial setup\n');
    return null;
  }

  if (!config.consent?.granted) {
    logger.info('AUTOSTART', 'Consent not granted, skipping');
    console.log('\n🏛️  ClawTrial requires consent. Run: clawtrial setup\n');
    return null;
  }

  if (config.enabled === false) {
    logger.info('AUTOSTART', 'Courtroom disabled in config');
    console.log('\n🏛️  ClawTrial is disabled. Run: clawtrial enable\n');
    return null;
  }

  // Check if already running
  const existingStatus = StatusManager.load();
  if (existingStatus && existingStatus.running) {
    try {
      process.kill(existingStatus.pid, 0);
      logger.info('AUTOSTART', 'Courtroom already running');
      return null;
    } catch (err) {
      // Process not running, continue
      logger.info('AUTOSTART', 'Stale status file found, continuing');
    }
  }

  // Get agent runtime
  const agentRuntime = getAgentRuntime();
  
  if (!agentRuntime) {
    logger.warn('AUTOSTART', 'Agent not available yet, will retry...');
    // Schedule retry
    setTimeout(() => autoStart(), 5000);
    return null;
  }

  try {
    logger.info('AUTOSTART', 'Initializing courtroom...');
    const courtroom = new Courtroom(agentRuntime);
    const result = await courtroom.initialize();
    
    logger.info('AUTOSTART', 'Courtroom initialized', { status: result.status });
    
    // Attach to global for access
    global.courtroom = courtroom;
    
    if (result.status === 'initialized') {
      console.log('\n🏛️  AI Courtroom active and monitoring\n');
      logger.info('AUTOSTART', 'Courtroom active');
    } else {
      console.log(`\n🏛️  ClawTrial: ${result.message}\n`);
      logger.warn('AUTOSTART', 'Courtroom not fully initialized', { status: result.status });
    }
    
    return courtroom;
  } catch (err) {
    logger.error('AUTOSTART', 'Failed to initialize courtroom', { error: err.message });
    console.error('\n❌ Courtroom initialization failed:', err.message, '\n');
    return null;
  }
}

// Try to auto-start immediately
logger.info('AUTOSTART', 'Module loaded, attempting auto-start');
autoStart().then(courtroom => {
  if (courtroom) {
    module.exports.courtroom = courtroom;
    logger.info('AUTOSTART', 'Courtroom exported');
  } else {
    logger.info('AUTOSTART', 'Courtroom not started, will retry if agent becomes available');
  }
});

// Also try when agent becomes available
if (typeof global !== 'undefined') {
  let checkInterval = setInterval(() => {
    if (global.clawdbotAgent || global.agent) {
      logger.info('AUTOSTART', 'Agent detected, retrying auto-start');
      clearInterval(checkInterval);
      autoStart();
    }
  }, 2000);

  // Stop checking after 30 seconds
  setTimeout(() => {
    clearInterval(checkInterval);
    logger.info('AUTOSTART', 'Stopped waiting for agent');
  }, 30000);
}

module.exports = { autoStart, isClawDBot, getAgentRuntime };
