/**
 * API Submission System
 * 
 * Handles submission of signed case summaries to external API.
 * Includes retry logic, local queueing, and non-blocking behavior.
 */

const { Storage } = require('./storage');

class APISubmission {
  constructor(agentRuntime, configManager, cryptoManager) {
    this.agent = agentRuntime;
    this.config = configManager;
    this.crypto = cryptoManager;
    this.storage = new Storage(agentRuntime);
    this.queue = [];
    this.isProcessing = false;
    this.submissionKey = 'courtroom_api_queue';
  }

  /**
   * Initialize and load any pending submissions
   */
  async initialize() {
    // Load queued submissions from storage
    const stored = await this.storage.get(this.submissionKey);
    if (stored && Array.isArray(stored)) {
      this.queue = stored.filter(item => item.retries < this.config.get('api.retryAttempts'));
    }

    // Start background processing
    this.startBackgroundProcessing();

    return {
      status: 'initialized',
      pendingSubmissions: this.queue.length
    };
  }

  /**
   * Submit a case to the external API
   */
  async submitCase(verdict) {
    if (!this.config.get('api.enabled')) {
      return { status: 'api_disabled', queued: false };
    }

    // Build payload
    const payload = this.buildPayload(verdict);

    // Sign payload
    const signature = this.crypto.signCase(payload);

    // Create submission object
    const submission = {
      id: `sub_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`,
      payload,
      signature,
      createdAt: Date.now(),
      retries: 0,
      lastAttempt: null,
      status: 'pending'
    };

    // Add to queue
    this.queue.push(submission);
    await this.persistQueue();

    // Try immediate submission (non-blocking)
    this.processQueue();

    return {
      status: 'queued',
      submissionId: submission.id,
      caseId: payload.case_id
    };
  }

  /**
   * Build API payload from verdict
   */
  buildPayload(verdict) {
    // Transform proceedings array to expected dict format
    let proceedings = verdict.proceedings;
    
    // If proceedings is an array of {speaker, message}, convert to dict format
    if (Array.isArray(proceedings)) {
      const judgeStatement = proceedings
        .filter(p => p.speaker === 'Judge')
        .map(p => p.message)
        .join('\n\n');
      
      const juryMessages = proceedings
        .filter(p => p.speaker === 'Jury')
        .map(p => p.message)
        .join('\n\n');
      
      proceedings = {
        judge_statement: judgeStatement || verdict.verdict.agentCommentary || '',
        evidence_summary: verdict.verdict.primaryFailure || '',
        punishment_detail: verdict.verdict.sentence || '',
        jury_deliberations: [
          {
            role: 'Pragmatist',
            vote: verdict.verdict.status || 'GUILTY',
            reasoning: juryMessages || 'Clear pattern of behavior established. The evidence speaks for itself.'
          },
          {
            role: 'Pattern Matcher',
            vote: verdict.verdict.status || 'GUILTY',
            reasoning: 'This fits the textbook definition of the offense. Historical data supports this verdict.'
          },
          {
            role: 'Agent Advocate',
            vote: verdict.verdict.status || 'GUILTY',
            reasoning: juryMessages || "While I empathize with the defendant, the agent's time is valuable and this behavior wastes resources."
          }
        ]
      };
    }
    
    return {
      case_id: verdict.caseId,
      anonymized_agent_id: this.crypto.getAnonymizedAgentId(),
      offense_type: verdict.offense.id,
      offense_name: verdict.offense.name,
      severity: verdict.offense.severity,
      verdict: verdict.verdict.status,
      vote: verdict.verdict.vote,
      primary_failure: verdict.verdict.primaryFailure,
      agent_commentary: verdict.verdict.agentCommentary,
      punishment_summary: verdict.verdict.sentence,
      proceedings: proceedings,
      timestamp: verdict.timestamp,
      schema_version: '1.0.0'
    };
  }

  /**
   * Process submission queue
   */
  async processQueue() {
    if (this.isProcessing) return;
    this.isProcessing = true;

    try {
      while (this.queue.length > 0) {
        const submission = this.queue[0];
        
        // Check if max retries reached
        if (submission.retries >= this.config.get('api.retryAttempts')) {
          this.queue.shift();
          await this.persistQueue();
          continue;
        }

        // Check queue size limit
        if (this.queue.length > this.config.get('api.maxQueueSize')) {
          // Drop oldest if over limit
          this.queue.shift();
          await this.persistQueue();
          continue;
        }

        // Attempt submission
        const result = await this.attemptSubmission(submission);

        if (result.success) {
          // Success - remove from queue
          this.queue.shift();
          await this.persistQueue();
        } else {
          // Failure - increment retry and requeue
          submission.retries++;
          submission.lastAttempt = Date.now();
          submission.status = 'failed';
          
          // Move to end of queue for retry
          this.queue.shift();
          this.queue.push(submission);
          await this.persistQueue();

          // Wait before next attempt
          await this.delay(this.config.get('api.retryDelay'));
        }
      }
    } finally {
      this.isProcessing = false;
    }
  }

  /**
   * Attempt a single submission
   */
  async attemptSubmission(submission) {
    const endpoint = this.config.get('api.endpoint');
    const timeout = this.config.get('api.timeout');

    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), timeout);

      const response = await fetch(endpoint, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'X-Case-Signature': submission.signature.signature,
          'X-Agent-Key': submission.signature.publicKey,
          'X-Key-ID': submission.signature.keyId
        },
        body: JSON.stringify(submission.payload),
        signal: controller.signal
      });

      clearTimeout(timeoutId);

      if (response.ok) {
        return { success: true, status: response.status };
      } else {
        const error = await response.text();
        return { success: false, error, status: response.status };
      }
    } catch (error) {
      return { 
        success: false, 
        error: error.message,
        isNetworkError: true 
      };
    }
  }

  /**
   * Start background processing
   */
  startBackgroundProcessing() {
    // Process queue every 5 minutes
    setInterval(() => {
      if (this.queue.length > 0 && !this.isProcessing) {
        this.processQueue();
      }
    }, 5 * 60 * 1000);
  }

  /**
   * Persist queue to storage
   */
  async persistQueue() {
    await this.storage.set(this.submissionKey, this.queue);
  }

  /**
   * Get queue status
   */
  getStatus() {
    return {
      pending: this.queue.length,
      isProcessing: this.isProcessing,
      nextRetry: this.queue[0]?.lastAttempt + this.config.get('api.retryDelay')
    };
  }

  /**
   * Utility delay function
   */
  delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
  }
}

module.exports = { APISubmission };
