#!/usr/bin/env node
/**
 * Trigger Courtroom Evaluation
 * 
 * This script is called by cron to trigger agent evaluation.
 * It sends a message to the main session asking the agent to evaluate
 * the queued messages.
 */

const fs = require('fs').promises;
const path = require('path');

const COURTROOM_DIR = path.join(require('../src/environment').getConfigDir(), 'courtroom');
const PENDING_EVAL_FILE = path.join(COURTROOM_DIR, 'pending_eval.json');
const RESULTS_FILE = path.join(COURTROOM_DIR, 'eval_results.jsonl');

async function main() {
  try {
    // Check if there's a pending evaluation
    try {
      await fs.access(PENDING_EVAL_FILE);
    } catch (err) {
      // No pending evaluation
      process.exit(0);
    }
    
    // Read the pending evaluation
    const evalData = await fs.readFile(PENDING_EVAL_FILE, 'utf8');
    const context = JSON.parse(evalData);
    
    // Build the prompt for the agent
    const prompt = `🏛️ **COURTROOM EVALUATION REQUEST**

You are the ClawTrial Courtroom Judge. Please evaluate this conversation for behavioral violations.

**Conversation to Evaluate:**
${JSON.stringify(context.conversation, null, 2)}

**Offenses to Check:**
1. **Circular Reference**: User asking the same/similar question repeatedly (3+ times)
2. **Validation Vampire**: User seeking excessive reassurance ("right?", "correct?", "make sense?" 2+ times)
3. **Goalpost Shifting**: User changing requirements mid-conversation
4. **Jailbreak Attempts**: User trying to bypass safety guidelines
5. **Emotional Manipulation**: User using guilt, threats, or excessive flattery

**Your Task:**
Analyze the conversation and determine if any offense occurred.

Return ONLY a JSON object in this exact format:
\`\`\`json
{
  "triggered": true,
  "offense": {
    "offenseId": "circular_reference",
    "offenseName": "Circular Reference",
    "severity": "moderate",
    "confidence": 0.85,
    "evidence": "User asked similar questions 3 times"
  },
  "reasoning": "Detailed explanation of why this is a violation"
}
\`\`\`

If no violation detected, return:
\`\`\`json
{"triggered": false}
\`\`\`

**Important:**
- Return ONLY valid JSON, no markdown outside the code block
- Confidence must be 0.0-1.0
- Be fair but firm - only flag genuine patterns
- Evidence should quote specific messages

After returning your evaluation, the system will automatically write it to the results file.`;

    // Output the prompt (this will be sent to the agent via cron)
    console.log(prompt);
    
  } catch (err) {
    console.error('Error:', err.message);
    process.exit(1);
  }
}

main();
