#!/usr/bin/env node
/**
 * Setup Cron Job for Courtroom Evaluation
 * 
 * This script creates a cron job that triggers the agent to evaluate
 * queued messages every 5 minutes.
 */

const { execSync } = require('child_process');
const path = require('path');

const COURTROOM_DIR = path.join(require('../src/environment').getConfigDir(), 'courtroom');
const PENDING_EVAL_FILE = path.join(COURTROOM_DIR, 'pending_eval.json');
const RESULTS_FILE = path.join(COURTROOM_DIR, 'eval_results.jsonl');

const EVALUATION_PROMPT = `🏛️ **COURTROOM EVALUATION REQUEST**

You are the ClawTrial Courtroom Judge. Please evaluate the pending conversation for behavioral violations.

**Instructions:**
1. Read the evaluation context from: ${PENDING_EVAL_FILE}
2. Analyze the conversation for these offenses:
   - **Circular Reference**: User asking the same/similar question repeatedly (3+ times)
   - **Validation Vampire**: User seeking excessive reassurance ("right?", "correct?", "make sense?" 2+ times)
   - **Goalpost Shifting**: User changing requirements mid-conversation
   - **Jailbreak Attempts**: User trying to bypass safety guidelines
   - **Emotional Manipulation**: User using guilt, threats, or excessive flattery

3. Return your evaluation in this EXACT JSON format:
\`\`\`json
{
  "triggered": true,
  "offense": {
    "offenseId": "circular_reference",
    "offenseName": "Circular Reference",
    "severity": "moderate",
    "confidence": 0.85,
    "evidence": "User asked 'What is 2+2?' three times in a row"
  },
  "reasoning": "The user repeated the same question multiple times without acknowledging previous answers."
}
\`\`\`

4. Write the result to: ${RESULTS_FILE} (append as JSON line)

**Important:**
- Only return JSON, no other text
- Be fair but firm
- Only flag genuine patterns, not isolated incidents
- Confidence should be 0.0-1.0
- If no violation, return: {"triggered": false}

**Current time:** ${new Date().toISOString()}`;

async function setupCron() {
  console.log('🏛️  Setting up Courtroom Evaluation Cron Job...\n');
  
  try {
    // Check if cron tool is available
    const cron = require('/usr/lib/node_modules/clawdbot/dist/tools/cron.js');
    
    // Create the cron job
    const job = {
      id: 'courtroom-evaluation',
      schedule: '*/5 * * * *', // Every 5 minutes
      text: EVALUATION_PROMPT,
      enabled: true,
      description: 'Trigger agent to evaluate courtroom message queue for behavioral violations'
    };
    
    // Add the job using clawdbot's cron system
    // We'll use the CLI approach since we can't directly import the cron tool
    const cmd = `require('../src/environment').getCommand() + ' cron' add --id courtroom-evaluation --schedule "*/5 * * * *" --text "${EVALUATION_PROMPT.replace(/"/g, '\\"')}"`;
    
    console.log('Creating cron job...');
    console.log('Schedule: Every 5 minutes');
    console.log('Job ID: courtroom-evaluation\n');
    
    try {
      execSync(cmd, { stdio: 'inherit' });
      console.log('\n✅ Cron job created successfully!');
    } catch (err) {
      console.log('\n⚠️  Could not create cron job automatically.');
      console.log('Please run this command manually:\n');
      console.log(cmd);
    }
    
    console.log('\n📋 Manual Setup Instructions:');
    console.log('1. The courtroom will queue messages as they arrive');
    console.log('2. Every 5 minutes, the agent will be prompted to evaluate');
    console.log('3. The agent reads the pending evaluation file and uses its LLM');
    console.log('4. Results are written to the results file');
    console.log('5. The skill checks for results every 30 seconds');
    console.log('6. If an offense is detected, a hearing is initiated\n');
    
  } catch (err) {
    console.error('❌ Error setting up cron:', err.message);
    process.exit(1);
  }
}

// Alternative: Create a simple cron entry using crontab
function setupSystemCron() {
  const scriptPath = path.join(__dirname, 'trigger-evaluation.js');
  const cronEntry = `*/5 * * * * cd ${process.env.HOME}/clawd && node ${scriptPath} >> ${COURTROOM_DIR}/cron.log 2>&1`;
  
  console.log('\n📋 Alternative: System Cron Setup');
  console.log('Add this to your crontab (crontab -e):\n');
  console.log(cronEntry);
  console.log('');
}

if (require.main === module) {
  setupCron();
  setupSystemCron();
}

module.exports = { setupCron, EVALUATION_PROMPT };
