#!/usr/bin/env node

/**
 * Post-install script for ClawTrial
 * Automatically configures the skill for the detected bot
 * 
 * Handles both:
 * - NPM install: Links from npm global to bot's skills directory
 * - ClawHub install: Links from workspace/skills to skills directory + creates CLI
 */

const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');

console.log('🏛️  ClawTrial Post-Install');

// Get package paths
const packagePath = path.join(__dirname, '..');
const cliSourcePath = path.join(packagePath, 'scripts', 'clawtrial.js');

// Detect which bot is installed
const homeDir = process.env.HOME || process.env.USERPROFILE || '';

const bots = [
  { name: 'openclaw', dir: '.openclaw', config: 'openclaw.json' },
  { name: 'moltbot', dir: '.moltbot', config: 'moltbot.json' },
  { name: 'clawdbot', dir: '.clawdbot', config: 'clawdbot.json' }
];

let detectedBot = null;

// Check which bot config exists
for (const bot of bots) {
  const configPath = path.join(homeDir, bot.dir, bot.config);
  if (fs.existsSync(configPath)) {
    detectedBot = bot;
    break;
  }
}

// Check if we're in ClawHub workspace or npm global
const isClawHubInstall = packagePath.includes('workspace/skills') || 
                         packagePath.includes('.openclaw/workspace');
const isNpmInstall = packagePath.includes('node_modules');

// Create CLI symlink in a location that's in PATH
function createCliSymlink() {
  const possiblePaths = [
    path.join(homeDir, '.npm-global', 'bin', 'clawtrial'),
    path.join(homeDir, '.local', 'bin', 'clawtrial'),
    '/usr/local/bin/clawtrial'
  ];
  
  // Find which directory exists and is writable
  for (const cliPath of possiblePaths) {
    try {
      const dir = path.dirname(cliPath);
      
      // Create directory if needed
      if (!fs.existsSync(dir)) {
        fs.mkdirSync(dir, { recursive: true });
      }
      
      // Remove old link if exists
      if (fs.existsSync(cliPath)) {
        fs.unlinkSync(cliPath);
      }
      
      // Create symlink
      fs.symlinkSync(cliSourcePath, cliPath);
      fs.chmodSync(cliSourcePath, 0o755);
      
      console.log(`✓ CLI installed: ${cliPath}`);
      return true;
    } catch (err) {
      // Try next path
      continue;
    }
  }
  
  console.log('⚠️  Could not create global CLI symlink');
  console.log('   You can still use: node ' + cliSourcePath);
  return false;
}

// Create CLI symlink
console.log('');
console.log('🔗 Installing CLI...');
createCliSymlink();

// Auto-link skill if bot detected
if (detectedBot) {
  console.log(`✓ Detected: ${detectedBot.name}`);
  
  const botDir = path.join(homeDir, detectedBot.dir);
  const skillsDir = path.join(botDir, 'skills');
  const skillLinkPath = path.join(skillsDir, 'clawtrial');
  
  try {
    // Create skills directory if needed
    if (!fs.existsSync(skillsDir)) {
      fs.mkdirSync(skillsDir, { recursive: true });
      console.log(`✓ Created skills directory: ${skillsDir}`);
    }
    
    // Remove old link if exists
    if (fs.existsSync(skillLinkPath)) {
      try { fs.unlinkSync(skillLinkPath); } catch (e) {}
    }
    
    // Create symlink
    fs.symlinkSync(packagePath, skillLinkPath, 'dir');
    console.log(`✓ Linked skill: ${skillLinkPath}`);
    
    if (isClawHubInstall) {
      console.log('  (Installed via ClawHub)');
    } else if (isNpmInstall) {
      console.log('  (Installed via NPM)');
    }
    
    // Enable in bot config if OpenClaw
    if (detectedBot.name === 'openclaw') {
      try {
        const botConfigPath = path.join(botDir, detectedBot.config);
        if (fs.existsSync(botConfigPath)) {
          const botConfig = JSON.parse(fs.readFileSync(botConfigPath, 'utf8'));
          
          if (!botConfig.skills) botConfig.skills = {};
          if (!botConfig.skills.entries) botConfig.skills.entries = {};
          botConfig.skills.entries.clawtrial = { enabled: true };
          
          fs.writeFileSync(botConfigPath, JSON.stringify(botConfig, null, 2));
          console.log('✓ Enabled in OpenClaw config');
        }
      } catch (configErr) {
        console.log(`⚠️  Could not update config: ${configErr.message}`);
      }
    }
    
    console.log('');
    console.log('🎉 ClawTrial is ready!');
    console.log('');
    console.log('Next step:');
    console.log(`  Restart ${detectedBot.name}: killall ${detectedBot.name} && ${detectedBot.name}`);
    
  } catch (err) {
    console.log(`⚠️  Could not link skill: ${err.message}`);
  }
} else {
  console.log('ℹ️  No bot detected');
}

console.log('');
console.log('📋 CLI Commands:');
console.log('  clawtrial status     - Check status');
console.log('  clawtrial setup      - Run setup');
console.log('  clawtrial diagnose   - Run diagnostics');
console.log('');
