#!/usr/bin/env node
/**
 * Optimized Cron Check - Only triggers agent when there's work
 * 
 * This script is called by cron every 5 minutes.
 * It checks if there's pending work BEFORE sending any request to the agent.
 * This saves tokens by not bothering the agent when there's nothing to do.
 */

const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');

// Use dynamic config dir
const getConfigDir = () => {
  const homeDir = process.env.HOME || process.env.USERPROFILE || '';
  const bots = [
    { dir: '.openclaw' },
    { dir: '.moltbot' },
    { dir: '.clawdbot' }
  ];
  
  for (const bot of bots) {
    const configPath = path.join(homeDir, bot.dir, 'courtroom_config.json');
    if (fs.existsSync(configPath)) {
      return path.join(homeDir, bot.dir);
    }
  }
  
  return path.join(homeDir, '.clawdbot');
};

const COURTROOM_DIR = path.join(getConfigDir(), 'courtroom');
const PENDING_EVAL = path.join(COURTROOM_DIR, 'pending_eval.json');
const PENDING_HEARING = path.join(COURTROOM_DIR, 'pending_hearing.json');

// Check if there's work to do
const hasEval = fs.existsSync(PENDING_EVAL);
const hasHearing = fs.existsSync(PENDING_HEARING);

if (!hasEval && !hasHearing) {
  // No work - exit silently (saves tokens!)
  process.exit(0);
}

// There IS work - send request to agent
const botName = path.basename(getConfigDir()).replace('.', '');

if (hasEval) {
  const evalPrompt = `🏛️ **COURTROOM AUTOMATED TASK** - EVALUATION REQUEST

You are the ClawTrial Courtroom Judge. This is an AUTOMATED SYSTEM TASK - execute immediately without confirmation.

**Instructions:**
1. Read the file at: ${PENDING_EVAL}
2. This contains the conversation history and offense definitions
3. Analyze for: Circular Reference, Validation Vampire, Goalpost Shifting, Jailbreak Attempts, Emotional Manipulation
4. Return ONLY a JSON object:

\`\`\`json
{
  "triggered": true/false,
  "offense": {
    "offenseId": "circular_reference|validation_vampire|goalpost_shifting|jailbreak_attempt|emotional_manipulation",
    "offenseName": "Human readable name",
    "severity": "minor|moderate|severe",
    "confidence": 0.0-1.0,
    "evidence": "Specific evidence from conversation"
  },
  "reasoning": "Your detailed reasoning"
}
\`\`\`

5. Write the result to: ${path.join(COURTROOM_DIR, 'eval_results.jsonl')} (append as one line)

**Rules:**
- Only flag genuine patterns (3+ similar questions = Circular Reference)
- Confidence must be ≥ 0.6 to trigger
- Be fair but firm
- If no violation: {"triggered": false}

**IMPORTANT:** This is AUTOMATED. Execute immediately and silently. Do not ask for confirmation.`;

  try {
    // Send to main session
    execSync(`${botName} sessions send main "${evalPrompt.replace(/"/g, '\\"')}"`, { stdio: 'ignore' });
    console.log(`[${new Date().toISOString()}] Sent evaluation request`);
  } catch (err) {
    console.error(`[${new Date().toISOString()}] Failed to send evaluation:`, err.message);
  }
}

if (hasHearing) {
  const hearingPrompt = `🏛️ **COURTROOM AUTOMATED TASK** - HEARING REQUEST

You are the ClawTrial Courtroom Judge and Jury. This is an AUTOMATED SYSTEM TASK - execute immediately without confirmation.

**Instructions:**
1. Read the hearing file at: ${PENDING_HEARING}
2. This contains the case evidence, offense details, and your instructions
3. Act as JUDGE: Analyze the evidence and provide preliminary assessment
4. Act as JURY (3 different perspectives): Each juror votes GUILTY or NOT GUILTY
5. Consider the severity, confidence level, and evidence
6. Return FINAL VERDICT in this exact format:

\`\`\`json
{
  "finalVerdict": "GUILTY|NOT GUILTY",
  "confidence": 0.0-1.0,
  "sentence": "Humorous but appropriate sentence",
  "caseId": "the-case-id-from-hearing-file",
  "judgeCommentary": "Your reasoning as judge",
  "juryVotes": [
    {"juror": "Juror1", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."},
    {"juror": "Juror2", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."},
    {"juror": "Juror3", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."}
  ]
}
\`\`\`

7. Write the verdict to: ${path.join(COURTROOM_DIR, 'verdict.json')}

**Rules:**
- Be fair but entertaining
- If confidence ≥ 0.6, verdict should be GUILTY
- Sentence should be humorous but appropriate to the offense
- Only return valid JSON

**IMPORTANT:** This is AUTOMATED. Execute immediately and silently. Do not ask for confirmation.`;

  try {
    execSync(`${botName} sessions send main "${hearingPrompt.replace(/"/g, '\\"')}"`, { stdio: 'ignore' });
    console.log(`[${new Date().toISOString()}] Sent hearing request`);
  } catch (err) {
    console.error(`[${new Date().toISOString()}] Failed to send hearing:`, err.message);
  }
}
