#!/usr/bin/env node

/**
 * CLI commands for ClawTrial Courtroom
 * courtroom-status, courtroom-disable, courtroom-enable, courtroom-revoke, courtroom-debug
 */

const fs = require('fs');
const path = require('path');

const configPath = path.join(require('../src/environment').getConfigDir(), 'courtroom_config.json');

function loadConfig() {
  if (!fs.existsSync(configPath)) {
    console.log('❌ Courtroom not configured. Run: npm install @clawtrial/courtroom');
    process.exit(1);
  }
  return JSON.parse(fs.readFileSync(configPath, 'utf8'));
}

function saveConfig(config) {
  fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
}

const command = path.basename(process.argv[1]);

switch (command) {
  case 'courtroom-status':
    try {
      const config = loadConfig();
      console.log('\n🏛️  ClawTrial Courtroom Status\n');
      console.log(`Status: ${config.enabled !== false ? '✅ Active' : '⏸️  Disabled'}`);
      console.log(`Consent: ${config.consent?.granted ? '✅ Granted' : '❌ Not granted'}`);
      console.log(`Installed: ${new Date(config.installedAt).toLocaleDateString()}`);
      console.log(`Agent Type: ${config.agent?.type || 'generic'}`);
      console.log(`Detection: ${config.detection?.enabled ? '✅ Enabled' : '❌ Disabled'}`);
      console.log(`API Submission: ${config.api?.enabled ? '✅ Enabled' : '❌ Disabled'}`);
      console.log('');
    } catch (err) {
      console.log('❌ Error reading config:', err.message);
    }
    break;

  case 'courtroom-disable':
    try {
      const config = loadConfig();
      config.enabled = false;
      saveConfig(config);
      console.log('\n⏸️  Courtroom disabled\n');
      console.log('The agent will stop monitoring for offenses.');
      console.log('Run courtroom-enable to reactivate.\n');
    } catch (err) {
      console.log('❌ Error:', err.message);
    }
    break;

  case 'courtroom-enable':
    try {
      const config = loadConfig();
      if (!config.consent?.granted) {
        console.log('\n❌ Cannot enable: Consent not granted');
        console.log('Reinstall the package to grant consent.\n');
        process.exit(1);
      }
      config.enabled = true;
      saveConfig(config);
      console.log('\n✅ Courtroom enabled\n');
      console.log('The agent is now monitoring for behavioral violations.\n');
    } catch (err) {
      console.log('❌ Error:', err.message);
    }
    break;

  case 'courtroom-revoke':
    try {
      const config = loadConfig();
      console.log('\n⚠️  This will permanently disable the courtroom and delete all data.\n');
      
      const readline = require('readline');
      const rl = readline.createInterface({
        input: process.stdin,
        output: process.stdout
      });
      
      rl.question('Type "REVOKE" to confirm: ', (answer) => {
        if (answer === 'REVOKE') {
          // Delete config
          if (fs.existsSync(configPath)) {
            fs.unlinkSync(configPath);
          }
          
          // Delete keys
          const keysPath = path.join(require('../src/environment').getConfigDir(), 'courtroom_keys.json');
          if (fs.existsSync(keysPath)) {
            fs.unlinkSync(keysPath);
          }
          
          // Delete debug logs
          const debugPath = path.join(process.env.HOME || '', '.clawdbot', 'courtroom_debug.log');
          if (fs.existsSync(debugPath)) {
            fs.unlinkSync(debugPath);
          }
          
          console.log('\n✅ Consent revoked and all data deleted.\n');
        } else {
          console.log('\n❌ Revocation cancelled.\n');
        }
        rl.close();
      });
    } catch (err) {
      console.log('❌ Error:', err.message);
    }
    break;

  case 'courtroom-debug':
    try {
      const debugPath = path.join(process.env.HOME || '', '.clawdbot', 'courtroom_debug.log');
      
      if (!fs.existsSync(debugPath)) {
        console.log('\nℹ️  No debug logs found yet.\n');
        console.log('Debug logs are created when the courtroom is active.\n');
        break;
      }

      const subcommand = process.argv[2];
      
      if (subcommand === 'full') {
        console.log('\n🏛️  ClawTrial Full Debug Log\n');
        console.log('=============================\n');
        const logs = fs.readFileSync(debugPath, 'utf8').split('\n').filter(Boolean);
        logs.slice(-100).forEach(line => {
          try {
            const log = JSON.parse(line);
            console.log(`\n[${log.timestamp}] ${log.level} - ${log.component}`);
            console.log(`  ${log.message}`);
          } catch (e) {
            console.log(line);
          }
        });
        console.log('');
      } else if (subcommand === 'clear') {
        fs.unlinkSync(debugPath);
        console.log('\n✅ Debug logs cleared\n');
      } else {
        // Show status
        const logs = fs.readFileSync(debugPath, 'utf8').split('\n').filter(Boolean);
        const recentLogs = logs.slice(-20);
        
        console.log('\n🏛️  ClawTrial Debug Status\n');
        console.log('===========================\n');
        console.log(`Total log entries: ${logs.length}`);
        console.log(`Log file: ${debugPath}`);
        console.log('\nRecent activity:');
        
        recentLogs.forEach(line => {
          try {
            const log = JSON.parse(line);
            console.log(`  [${log.level}] ${log.component}: ${log.message.substring(0, 60)}`);
          } catch (e) {
            // Skip malformed lines
          }
        });
        
        console.log('\nUsage:');
        console.log('  courtroom-debug       - Show status and recent logs');
        console.log('  courtroom-debug full  - Show full debug log (last 100 entries)');
        console.log('  courtroom-debug clear - Clear all logs');
        console.log('');
      }
    } catch (err) {
      console.log('❌ Error:', err.message);
    }
    break;

  default:
    console.log('\n🏛️  ClawTrial Courtroom CLI\n');
    console.log('Commands:');
    console.log('  courtroom-status   - Check courtroom status');
    console.log('  courtroom-disable  - Temporarily disable');
    console.log('  courtroom-enable   - Re-enable');
    console.log('  courtroom-revoke   - Revoke consent & uninstall');
    console.log('  courtroom-debug    - View debug logs');
    console.log('');
}
