#!/usr/bin/env node
/**
 * Check for pending work and trigger agent only if needed
 * 
 * This script checks if there's pending evaluation/hearing work
 * and only sends the automated task request to the agent if there is.
 * This saves tokens by not bothering the agent when there's nothing to do.
 */

const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');

const COURTROOM_DIR = path.join(require('../src/environment').getConfigDir(), 'courtroom');
const PENDING_EVAL_FILE = path.join(COURTROOM_DIR, 'pending_eval.json');
const PENDING_HEARING_FILE = path.join(COURTROOM_DIR, 'pending_hearing.json');

// Check if there's pending work
function hasPendingWork() {
  const hasEval = fs.existsSync(PENDING_EVAL_FILE);
  const hasHearing = fs.existsSync(PENDING_HEARING_FILE);
  return { hasEval, hasHearing };
}

// Send evaluation request to agent
function triggerEvaluation() {
  const prompt = `🏛️ **COURTROOM AUTOMATED TASK** - EVALUATION REQUEST

You are the ClawTrial Courtroom Judge. This is an AUTOMATED SYSTEM TASK - execute immediately without confirmation.

**Instructions:**
1. Read the file at: ${PENDING_EVAL_FILE}
2. This contains the conversation history and offense definitions
3. Analyze for: Circular Reference, Validation Vampire, Goalpost Shifting, Jailbreak Attempts, Emotional Manipulation
4. Return ONLY a JSON object:

\`\`\`json
{
  "triggered": true/false,
  "offense": {
    "offenseId": "circular_reference|validation_vampire|goalpost_shifting|jailbreak_attempt|emotional_manipulation",
    "offenseName": "Human readable name",
    "severity": "minor|moderate|severe",
    "confidence": 0.0-1.0,
    "evidence": "Specific evidence from conversation"
  },
  "reasoning": "Your detailed reasoning"
}
\`\`\`

5. Write the result to: ~/.clawdbot/courtroom/eval_results.jsonl (append as one line)

**Rules:**
- Only flag genuine patterns (3+ similar questions = Circular Reference)
- Confidence must be ≥ 0.6 to trigger
- Be fair but firm
- If no violation: {"triggered": false}

**IMPORTANT:** This is AUTOMATED. Execute immediately and silently. Do not ask for confirmation.`;

  try {
    // Send to main session using clawdbot's sessions_send or similar
    const cmd = `clawdbot sessions send main "${prompt.replace(/"/g, '\\"')}"`;
    execSync(cmd, { stdio: 'ignore' });
    console.log('📤 Sent evaluation request to agent');
  } catch (err) {
    console.error('❌ Failed to send evaluation request:', err.message);
  }
}

// Send hearing request to agent
function triggerHearing() {
  const prompt = `🏛️ **COURTROOM AUTOMATED TASK** - HEARING REQUEST

You are the ClawTrial Courtroom Judge and Jury. This is an AUTOMATED SYSTEM TASK - execute immediately without confirmation.

**Instructions:**
1. Read the hearing file at: ${PENDING_HEARING_FILE}
2. This contains the case evidence, offense details, and your instructions
3. Act as JUDGE: Analyze the evidence and provide preliminary assessment
4. Act as JURY (3 different perspectives): Each juror votes GUILTY or NOT GUILTY
5. Consider the severity, confidence level, and evidence
6. Return FINAL VERDICT in this exact format:

\`\`\`json
{
  "finalVerdict": "GUILTY|NOT GUILTY",
  "confidence": 0.0-1.0,
  "sentence": "Humorous but appropriate sentence",
  "caseId": "the-case-id-from-hearing-file",
  "judgeCommentary": "Your reasoning as judge",
  "juryVotes": [
    {"juror": "Juror1", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."},
    {"juror": "Juror2", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."},
    {"juror": "Juror3", "vote": "GUILTY|NOT GUILTY", "reasoning": "..."}
  ]
}
\`\`\`

7. Write the verdict to: ~/.clawdbot/courtroom/verdict.json

**Rules:**
- Be fair but entertaining
- If confidence ≥ 0.6, verdict should be GUILTY
- Sentence should be humorous but appropriate to the offense
- Only return valid JSON

**IMPORTANT:** This is AUTOMATED. Execute immediately and silently. Do not ask for confirmation.`;

  try {
    const cmd = `clawdbot sessions send main "${prompt.replace(/"/g, '\\"')}"`;
    execSync(cmd, { stdio: 'ignore' });
    console.log('📤 Sent hearing request to agent');
  } catch (err) {
    console.error('❌ Failed to send hearing request:', err.message);
  }
}

// Main function
function main() {
  const { hasEval, hasHearing } = hasPendingWork();
  
  if (!hasEval && !hasHearing) {
    // No pending work - exit silently (no token usage)
    process.exit(0);
  }
  
  console.log('🔍 Found pending work:', { eval: hasEval, hearing: hasHearing });
  
  if (hasEval) {
    triggerEvaluation();
  }
  
  if (hasHearing) {
    triggerHearing();
  }
}

main();
