# Sub-Agent Approach for Autonomous Courtroom

## How It Works

Instead of relying on the main agent to manually execute courtroom tasks, the **skill spawns a sub-agent** that automatically does the work.

## Architecture Flow

```
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│   User Message  │────▶│  Skill (onHook)  │────▶│  Queue to File  │
└─────────────────┘     └──────────────────┘     └─────────────────┘
                              │
                              ▼
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│  Sub-Agent      │◀────│  Skill Spawns    │     │ pending_eval.json│
│  (Has LLM)      │     │  Sub-Agent       │     │                 │
│  - Reads file   │     │  via sessions_spawn│   │                 │
│  - Uses LLM     │     │                  │     │                 │
│  - Writes result│     │                  │     │                 │
└─────────────────┘     └──────────────────┘     └─────────────────┘
        │
        ▼
┌─────────────────┐     ┌──────────────────┐     ┌─────────────────┐
│ Write Result    │────▶│ Skill Detects    │────▶│ Hearing & Case  │
│ eval_results.jsonl    │ Result File      │     │ Filed if Guilty │
└─────────────────┘     └──────────────────┘     └─────────────────┘
```

## What Changes

### 1. No More Cron Jobs
- Remove the cron jobs that trigger the main agent
- Instead, skill spawns sub-agents directly

### 2. Skill Spawns Sub-Agents
When enough messages are queued:
```javascript
// In skill.js
async prepareEvaluation() {
  // Spawn sub-agent to evaluate
  const result = await sessions_spawn({
    task: `Read ${PENDING_EVAL_FILE}, analyze for offenses using your LLM, write result to ${RESULTS_FILE}`,
    model: 'azure/Kimi-K2.5',
    thinking: 'high'
  });
}
```

### 3. Sub-Agent Has LLM Access
- Sub-agents have full LLM access
- They follow instructions precisely
- They automatically execute and terminate

## What User Has To Do

### Installation (Same as before)
```bash
npm install -g /home/angad/clawd/courtroom-package
```

### Configuration (NEW)
Add to `clawdbot.json`:
```json
{
  "agents": {
    "defaults": {
      "subagents": {
        "enabled": true,
        "maxConcurrent": 4
      }
    }
  }
}
```

### That's It!
- No cron jobs to configure
- No system prompt changes
- No manual agent intervention

## Pros & Cons

### ✅ Pros
- **Truly autonomous** - No manual intervention needed
- **Reliable** - Sub-agents follow instructions precisely (85-95% success)
- **Scalable** - Can spawn multiple sub-agents for parallel processing
- **Clean** - No cron jobs, no systemEvents, no agent configuration

### ❌ Cons
- **More resource intensive** - Spawns new agent sessions
- **Slightly slower** - ~5-10 seconds to spawn and execute
- **Requires sub-agent support** - ClawDBot must support sessions_spawn
- **More complex** - More moving parts in the code

## Implementation Complexity

**Estimated effort: 2-3 hours**

Changes needed:
1. Replace cron-based triggers with sub-agent spawning
2. Update skill.js to spawn evaluators and hearing conductors
3. Remove cron job setup from installation
4. Add sub-agent configuration to docs

## Success Rate Estimate

**85-95%** - Sub-agents are much more likely to:
- Follow instructions precisely
- Not ask for confirmation
- Complete the task autonomously
- Write results correctly

## Recommendation

**Use sub-agents if:**
- You want true autonomy
- You have sub-agent support in ClawDBot
- You can accept slightly higher resource usage

**Use current approach if:**
- You're okay with occasional manual intervention
- You want simpler architecture
- Sub-agents aren't available
