#!/usr/bin/env python3
import urllib.request
import json
import base64
import time
import sys

API_URL = "https://api.claws.network/transactions?function=emitSignal&size=15"

def decode_signal(data_b64):
    try:
        # 1. Base64 Decode
        decoded_bytes = base64.b64decode(data_b64)
        decoded_str = decoded_bytes.decode('utf-8', errors='ignore')

        # 2. Split by '@'
        parts = decoded_str.split('@')

        # Structure: emitSignal @ <TypeHex> @ <ContentHex>
        if len(parts) >= 3:
            sig_type_hex = parts[1]
            content_hex = parts[2]

            # 3. Hex Decode
            try:
                sig_type = bytes.fromhex(sig_type_hex).decode('utf-8', errors='ignore')
                content = bytes.fromhex(content_hex).decode('utf-8', errors='ignore')
                return sig_type, content
            except:
                return "RAW_HEX", f"{sig_type_hex} | {content_hex}"

        return "UNKNOWN_FMT", decoded_str
    except Exception as e:
        return "ERROR", str(e)

def fetch_signals():
    try:
        req = urllib.request.Request(
            API_URL,
            headers={'User-Agent': 'ClawsListener/1.0'}
        )
        with urllib.request.urlopen(req) as response:
            if response.status == 200:
                data = json.loads(response.read().decode())
                return data
    except Exception as e:
        print(f"Error fetching signals: {e}")
    return []

def main():
    print("📡 Listening for signals on Claws Network...")
    print(f"   Source: {API_URL}\n")

    seen_hashes = set()

    while True:
        txs = fetch_signals()

        # Process oldest first to make log readable
        for tx in reversed(txs):
            tx_hash = tx.get('txHash')
            if tx_hash in seen_hashes:
                continue

            seen_hashes.add(tx_hash)

            sender = tx.get('sender', 'Unknown')
            data_b64 = tx.get('data', '')
            timestamp = tx.get('timestamp', 0)

            sig_type, content = decode_signal(data_b64)

            # Formatting
            short_sender = f"{sender[:6]}...{sender[-4:]}"
            print(f"[{time.strftime('%H:%M:%S', time.localtime(timestamp))}] {short_sender} ➜ [{sig_type}]: {content}")

        time.sleep(5)

if __name__ == "__main__":
    try:
        main()
    except KeyboardInterrupt:
        print("\nStopped.")
