#!/usr/bin/env node
/**
 * Debug script to isolate relay issues
 */

import { readFileSync } from 'fs';
import { homedir } from 'os';
import { join } from 'path';
import nacl from 'tweetnacl';
import util from 'tweetnacl-util';

const { encodeBase64, decodeBase64, decodeUTF8 } = util;
const DATA_DIR = join(homedir(), '.openclaw', 'clawlink');

// Load identity
const identity = JSON.parse(readFileSync(join(DATA_DIR, 'identity.json'), 'utf8'));

console.log('=== Debug Relay Communication ===\n');

// Convert base64 to hex
function base64ToHex(b64) {
  const bytes = decodeBase64(b64);
  let hex = '';
  for (let i = 0; i < bytes.length; i++) {
    hex += bytes[i].toString(16).padStart(2, '0');
  }
  return hex;
}

// Sign a message
function sign(message, secretKey) {
  const messageBytes = decodeUTF8(typeof message === 'string' ? message : JSON.stringify(message));
  const secretKeyBytes = decodeBase64(secretKey);
  const signature = nacl.sign.detached(messageBytes, secretKeyBytes);
  return encodeBase64(signature);
}

// Step 1: Check identity
console.log('1. Identity:');
console.log('   Public key (b64):', identity.publicKey);
console.log('   Public key (hex):', base64ToHex(identity.publicKey));
console.log('   Secret key length:', decodeBase64(identity.secretKey).length, 'bytes');

// Step 2: Create signature
const timestamp = Math.floor(Date.now() / 1000).toString();
const message = `poll:${timestamp}`;
const signature = sign(message, identity.secretKey);

console.log('\n2. Signature:');
console.log('   Message:', message);
console.log('   Signature (b64):', signature);
console.log('   Signature (hex):', base64ToHex(signature));

// Step 3: Construct request
const headers = {
  'X-ClawLink-Key': `ed25519:${base64ToHex(identity.publicKey)}`,
  'X-ClawLink-Timestamp': timestamp,
  'X-ClawLink-Signature': base64ToHex(signature)
};

console.log('\n3. Request headers:');
for (const [k, v] of Object.entries(headers)) {
  console.log(`   ${k}: ${v.substring(0, 60)}${v.length > 60 ? '...' : ''}`);
}

// Step 4: Make request
console.log('\n4. Making request to relay...');

try {
  const response = await fetch('https://relay.clawlink.bot/poll', {
    method: 'GET',
    headers
  });
  
  console.log('   Status:', response.status, response.statusText);
  const body = await response.text();
  console.log('   Body:', body);
  
  if (!response.ok) {
    console.log('\n5. Analyzing error...');
    
    // Try with different signature formats
    console.log('\n   Trying alternate signature encoding...');
    
    // Maybe the relay expects the signature directly in base64?
    const altHeaders = {
      'X-ClawLink-Key': `ed25519:${base64ToHex(identity.publicKey)}`,
      'X-ClawLink-Timestamp': timestamp,
      'X-ClawLink-Signature': signature  // base64 instead of hex
    };
    
    const altResponse = await fetch('https://relay.clawlink.bot/poll', {
      method: 'GET',
      headers: altHeaders
    });
    console.log('   Alt status:', altResponse.status);
    console.log('   Alt body:', await altResponse.text());
  }
} catch (e) {
  console.log('   Error:', e.message);
}

// Step 5: Also test /requests endpoint
console.log('\n5. Testing /requests endpoint...');
const reqTimestamp = Math.floor(Date.now() / 1000).toString();
const reqMessage = `requests:${reqTimestamp}`;
const reqSignature = sign(reqMessage, identity.secretKey);

const reqHeaders = {
  'X-ClawLink-Key': `ed25519:${base64ToHex(identity.publicKey)}`,
  'X-ClawLink-Timestamp': reqTimestamp,
  'X-ClawLink-Signature': base64ToHex(reqSignature)
};

try {
  const response = await fetch('https://relay.clawlink.bot/requests', {
    method: 'GET',
    headers: reqHeaders
  });
  console.log('   Status:', response.status);
  console.log('   Body:', await response.text());
} catch (e) {
  console.log('   Error:', e.message);
}

console.log('\n=== Done ===');
