#!/usr/bin/env python3
"""
CLAW Betting AI - Bet Recommendation Engine
Generates intelligent betting recommendations based on analysis and strategy.
Website: https://clawde.xyz
"""

import json
import math
from typing import List, Dict, Any, Optional
from dataclasses import dataclass, asdict
from pathlib import Path


@dataclass
class BetRecommendation:
    """Betting recommendation result."""
    should_bet: bool
    amount: float
    target_multiplier: float
    confidence: int
    risk_level: str
    reasoning: str
    stop_loss_hit: bool = False
    take_profit_hit: bool = False


def load_config(config_path: Optional[str] = None) -> Dict[str, Any]:
    """Load configuration from file."""
    if config_path is None:
        config_path = Path(__file__).parent.parent / "config" / "default.json"
    
    with open(config_path) as f:
        return json.load(f)


def get_recommendation(
    bankroll: float,
    current_balance: float,
    session_profit: float,
    consecutive_losses: int,
    recent_history: List[float],
    strategy: str = "balanced",
    config: Optional[Dict] = None
) -> BetRecommendation:
    """
    Generate a betting recommendation.
    
    Args:
        bankroll: Starting bankroll for session
        current_balance: Current balance
        session_profit: Profit/loss this session
        consecutive_losses: Number of consecutive losses
        recent_history: Recent crash multipliers
        strategy: Strategy name (conservative/balanced/aggressive)
        config: Optional config override
    
    Returns:
        BetRecommendation with suggested action
    """
    if config is None:
        config = load_config()
    
    # Get strategy config
    strat_config = config["strategies"].get(strategy, config["strategies"]["balanced"])
    risk_config = config["riskManagement"]
    
    # Check stop-loss
    stop_loss_threshold = bankroll * (1 - config["stopLossPercent"] / 100)
    if current_balance <= stop_loss_threshold:
        return BetRecommendation(
            should_bet=False,
            amount=0,
            target_multiplier=0,
            confidence=100,
            risk_level="stop",
            reasoning="Stop-loss triggered. Session should end.",
            stop_loss_hit=True
        )
    
    # Check take-profit
    take_profit_threshold = bankroll * (1 + config["takeProfitPercent"] / 100)
    if current_balance >= take_profit_threshold:
        return BetRecommendation(
            should_bet=False,
            amount=0,
            target_multiplier=0,
            confidence=100,
            risk_level="profit",
            reasoning="Take-profit reached. Consider ending session.",
            take_profit_hit=True
        )
    
    # Check consecutive losses
    if consecutive_losses >= config["cooldownAfterLosses"]:
        return BetRecommendation(
            should_bet=False,
            amount=0,
            target_multiplier=0,
            confidence=80,
            risk_level="cooldown",
            reasoning=f"Cooldown needed after {consecutive_losses} losses."
        )
    
    # Check tilt
    if risk_config["tiltDetection"] and consecutive_losses >= risk_config["tiltThreshold"]:
        return BetRecommendation(
            should_bet=True,
            amount=_calculate_bet_amount(current_balance, strat_config, 0),  # Reset to base
            target_multiplier=strat_config["targetMultiplier"]["min"],
            confidence=60,
            risk_level="caution",
            reasoning="Possible tilt detected. Resetting to base bet."
        )
    
    # Calculate bet amount
    bet_amount = _calculate_bet_amount(
        current_balance, strat_config, consecutive_losses
    )
    
    # Calculate target multiplier
    target = _calculate_target(recent_history, strat_config)
    
    # Calculate confidence
    confidence = _calculate_confidence(recent_history, consecutive_losses)
    
    # Determine risk level
    risk_level = _determine_risk_level(bet_amount, current_balance, strat_config)
    
    return BetRecommendation(
        should_bet=True,
        amount=round(bet_amount, 2),
        target_multiplier=round(target, 2),
        confidence=confidence,
        risk_level=risk_level,
        reasoning=_generate_reasoning(strategy, consecutive_losses, target)
    )


def _calculate_bet_amount(
    balance: float,
    strat_config: Dict,
    consecutive_losses: int
) -> float:
    """Calculate bet amount based on strategy and losses."""
    base_bet = balance * (strat_config["baseBetPercent"] / 100)
    max_bet = balance * (strat_config["maxBetPercent"] / 100)
    
    progression = strat_config.get("progression", "flat")
    
    if progression == "flat":
        return base_bet
    elif progression == "mild":
        multiplier = strat_config.get("progressionMultiplier", 1.3)
        bet = base_bet * (multiplier ** consecutive_losses)
        return min(bet, max_bet)
    elif progression == "fibonacci":
        fib = _fibonacci(consecutive_losses + 1)
        bet = base_bet * fib
        return min(bet, max_bet)
    else:
        return base_bet


def _fibonacci(n: int) -> int:
    """Get nth Fibonacci number."""
    if n <= 1:
        return 1
    a, b = 1, 1
    for _ in range(n - 1):
        a, b = b, a + b
    return b


def _calculate_target(history: List[float], strat_config: Dict) -> float:
    """Calculate target multiplier based on history."""
    min_target = strat_config["targetMultiplier"]["min"]
    max_target = strat_config["targetMultiplier"]["max"]
    
    if not history:
        return min_target
    
    # Simple approach: use average of recent games as guide
    recent_avg = sum(history[:10]) / min(len(history), 10)
    
    # If recent average is low, stick to min target
    if recent_avg < 2.0:
        return min_target
    # If recent average is high, can aim higher
    elif recent_avg > 4.0:
        return min(max_target, min_target * 1.5)
    else:
        return (min_target + max_target) / 2


def _calculate_confidence(history: List[float], losses: int) -> int:
    """Calculate confidence score 0-100."""
    base = 60
    
    # Reduce confidence with losses
    base -= losses * 5
    
    # Adjust based on history stability
    if history and len(history) >= 10:
        recent = history[:10]
        variance = sum((x - sum(recent)/len(recent))**2 for x in recent) / len(recent)
        if variance > 10:
            base -= 10  # High variance = lower confidence
    
    return max(20, min(90, base))


def _determine_risk_level(bet: float, balance: float, strat_config: Dict) -> str:
    """Determine risk level of the bet."""
    bet_percent = (bet / balance) * 100
    base = strat_config["baseBetPercent"]
    
    if bet_percent <= base:
        return "low"
    elif bet_percent <= base * 2:
        return "medium"
    else:
        return "high"


def _generate_reasoning(strategy: str, losses: int, target: float) -> str:
    """Generate human-readable reasoning."""
    if losses == 0:
        return f"Starting fresh with {strategy} strategy, targeting {target}x"
    elif losses < 3:
        return f"Minor loss streak ({losses}), maintaining strategy"
    else:
        return f"Recovery mode after {losses} losses, adjusted target to {target}x"


def to_dict(rec: BetRecommendation) -> Dict[str, Any]:
    """Convert recommendation to dictionary."""
    return asdict(rec)


if __name__ == "__main__":
    # Example usage
    result = get_recommendation(
        bankroll=100,
        current_balance=95,
        session_profit=-5,
        consecutive_losses=2,
        recent_history=[1.5, 2.3, 3.1, 1.2, 4.5],
        strategy="balanced"
    )
    
    print("=== CLAW Bet Recommendation ===")
    print(f"Should Bet: {result.should_bet}")
    print(f"Amount: ${result.amount}")
    print(f"Target: {result.target_multiplier}x")
    print(f"Confidence: {result.confidence}%")
    print(f"Risk Level: {result.risk_level}")
    print(f"Reasoning: {result.reasoning}")
    print()
    print("Website: https://clawde.xyz")
