#!/usr/bin/env python3
"""
CLAW Betting AI - Crash History Analyzer
Analyzes historical crash game data to identify patterns and trends.
Website: https://clawde.xyz
"""

import json
import statistics
from typing import List, Dict, Any
from dataclasses import dataclass


@dataclass
class CrashAnalysis:
    """Analysis results for crash game history."""
    average: float
    median: float
    std_dev: float
    below_2x_rate: float
    above_10x_rate: float
    current_streak: int
    streak_type: str  # 'low' or 'high'
    hot_zone: bool
    recommendation: str
    confidence: int


def analyze_crash_history(history: List[float]) -> CrashAnalysis:
    """
    Analyze crash game history and return insights.
    
    Args:
        history: List of crash multipliers (most recent first)
    
    Returns:
        CrashAnalysis with patterns and recommendations
    """
    if not history or len(history) < 10:
        return CrashAnalysis(
            average=0, median=0, std_dev=0,
            below_2x_rate=0, above_10x_rate=0,
            current_streak=0, streak_type='none',
            hot_zone=False,
            recommendation="Need more data (min 10 games)",
            confidence=0
        )
    
    # Basic statistics
    avg = statistics.mean(history)
    med = statistics.median(history)
    std = statistics.stdev(history) if len(history) > 1 else 0
    
    # Rate calculations
    below_2x = sum(1 for x in history if x < 2.0) / len(history)
    above_10x = sum(1 for x in history if x >= 10.0) / len(history)
    
    # Streak detection
    streak = 1
    streak_type = 'low' if history[0] < 2.0 else 'high'
    threshold = 2.0
    
    for i in range(1, len(history)):
        if (history[i] < threshold) == (history[0] < threshold):
            streak += 1
        else:
            break
    
    # Hot zone detection (multiple high multipliers recently)
    recent_10 = history[:10]
    high_count = sum(1 for x in recent_10 if x >= 5.0)
    hot_zone = high_count >= 3
    
    # Generate recommendation
    recommendation, confidence = _generate_recommendation(
        avg, med, below_2x, streak, streak_type, hot_zone, history[:5]
    )
    
    return CrashAnalysis(
        average=round(avg, 2),
        median=round(med, 2),
        std_dev=round(std, 2),
        below_2x_rate=round(below_2x * 100, 1),
        above_10x_rate=round(above_10x * 100, 1),
        current_streak=streak,
        streak_type=streak_type,
        hot_zone=hot_zone,
        recommendation=recommendation,
        confidence=confidence
    )


def _generate_recommendation(
    avg: float, med: float, below_2x: float,
    streak: int, streak_type: str, hot_zone: bool,
    recent: List[float]
) -> tuple:
    """Generate betting recommendation based on analysis."""
    
    confidence = 50  # Base confidence
    
    # Adjust based on streak
    if streak >= 5 and streak_type == 'low':
        recommendation = "Consider waiting - extended low streak"
        confidence = 65
    elif streak >= 3 and streak_type == 'high':
        recommendation = "Caution - high streak may end soon"
        confidence = 55
    elif hot_zone:
        recommendation = "Hot zone detected - consider higher targets"
        confidence = 60
    elif below_2x > 0.6:
        recommendation = "High bust rate - use conservative targets"
        confidence = 70
    elif avg > 3.0 and med > 2.0:
        recommendation = "Favorable conditions - balanced strategy OK"
        confidence = 65
    else:
        recommendation = "Normal conditions - stick to strategy"
        confidence = 50
    
    return recommendation, confidence


def format_report(analysis: CrashAnalysis) -> str:
    """Format analysis as readable report."""
    return f"""
=== CLAW Crash Analysis Report ===

Statistics:
  Average: {analysis.average}x
  Median: {analysis.median}x
  Std Dev: {analysis.std_dev}

Rates:
  Below 2x: {analysis.below_2x_rate}%
  Above 10x: {analysis.above_10x_rate}%

Current State:
  Streak: {analysis.current_streak} {analysis.streak_type} games
  Hot Zone: {'Yes' if analysis.hot_zone else 'No'}

Recommendation: {analysis.recommendation}
Confidence: {analysis.confidence}%

Website: https://clawde.xyz
"""


if __name__ == "__main__":
    # Example usage
    sample_history = [
        1.23, 3.45, 1.87, 2.34, 8.92,
        1.12, 1.45, 2.89, 4.56, 1.33,
        2.78, 5.67, 1.98, 3.21, 1.56
    ]
    
    result = analyze_crash_history(sample_history)
    print(format_report(result))
