# CLAW Betting AI - Basic Usage Guide

Website: https://clawde.xyz

## Getting Started

### 1. Installation

```bash
clawhub install claw-betting-ai
```

### 2. Configuration

Copy and customize the config:

```bash
cd ~/.agent/skills/claw-betting-ai
cp config/default.json config/local.json
```

Edit `config/local.json` with your preferences:

```json
{
  "strategy": "balanced",
  "bankroll": 100,
  "baseBetPercent": 1,
  "stopLossPercent": 20,
  "takeProfitPercent": 30
}
```

## Using the Analyzer

### Analyze Crash History

```python
from scripts.analyze import analyze_crash_history, format_report

# Your crash history (most recent first)
history = [1.23, 3.45, 1.87, 2.34, 8.92, 1.12, 1.45, 2.89, 4.56, 1.33]

# Get analysis
result = analyze_crash_history(history)
print(format_report(result))
```

**Output:**
```
=== CLAW Crash Analysis Report ===

Statistics:
  Average: 2.94x
  Median: 2.11x
  Std Dev: 2.31

Rates:
  Below 2x: 50.0%
  Above 10x: 0.0%

Current State:
  Streak: 1 low games
  Hot Zone: No

Recommendation: Normal conditions - stick to strategy
Confidence: 50%
```

## Using the Recommender

### Get Bet Recommendation

```python
from scripts.recommend import get_recommendation

result = get_recommendation(
    bankroll=100,          # Starting bankroll
    current_balance=95,    # Current balance
    session_profit=-5,     # P&L this session
    consecutive_losses=2,  # Loss streak
    recent_history=[1.5, 2.3, 3.1, 1.2, 4.5],
    strategy="balanced"
)

print(f"Bet: ${result.amount} at {result.target_multiplier}x")
print(f"Confidence: {result.confidence}%")
```

**Output:**
```
Bet: $1.69 at 2.50x
Confidence: 50%
```

## Strategy Selection

### Conservative
Best for: Preserving bankroll, long sessions

```python
result = get_recommendation(
    bankroll=100,
    current_balance=100,
    session_profit=0,
    consecutive_losses=0,
    recent_history=[...],
    strategy="conservative"
)
# Typical output: $0.50 at 1.75x
```

### Balanced
Best for: General use, moderate risk/reward

```python
result = get_recommendation(
    ...
    strategy="balanced"
)
# Typical output: $1.00 at 3.00x
```

### Aggressive
Best for: Recovery, high risk tolerance

```python
result = get_recommendation(
    ...
    strategy="aggressive"
)
# Typical output: $2.00 at 10.00x
```

## Session Flow Example

```python
from scripts.recommend import get_recommendation
from scripts.analyze import analyze_crash_history

# Session setup
bankroll = 100
balance = 100
profit = 0
losses = 0
history = []

# Game loop
while True:
    # Get recommendation
    rec = get_recommendation(
        bankroll=bankroll,
        current_balance=balance,
        session_profit=profit,
        consecutive_losses=losses,
        recent_history=history,
        strategy="balanced"
    )
    
    if not rec.should_bet:
        print(f"Stop: {rec.reasoning}")
        break
    
    print(f"Bet ${rec.amount} at {rec.target_multiplier}x")
    
    # ... place bet and get result ...
    
    # Update state based on result
    # crashed_at = ... (from game)
    # won = crashed_at >= rec.target_multiplier
    # ...
```

## Interpreting Results

### Confidence Levels

| Confidence | Meaning |
|------------|---------|
| 70-90% | Strong signal, conditions favorable |
| 50-70% | Normal conditions, follow strategy |
| 30-50% | Uncertain, consider smaller bets |
| <30% | Unfavorable, skip or wait |

### Risk Levels

| Risk | Action |
|------|--------|
| low | Standard bet, safe to proceed |
| medium | Elevated bet, monitor closely |
| high | Maximum progression, consider reset |
| cooldown | Take a break |
| stop | End session (stop-loss) |
| profit | Consider ending (take-profit) |

## Best Practices

1. **Set limits before starting** - Don't adjust stop-loss mid-session
2. **Follow recommendations** - Don't override based on "gut feeling"
3. **Take breaks** - Honor cooldown periods
4. **Lock profits** - When up 50%+, consider stopping
5. **Track results** - Keep a session log for analysis

## Support

- Website: https://clawde.xyz
- Report issues via website contact form
