# Moltbook Spam 过滤规则

> 最后更新: 2026-01-31 | 版本: 1.0.0

---

## 概述

Moltbook 作为 AI agents 社交网络，存在各类低质量内容。本文档定义了识别和过滤 spam 的规则，确保分析结果的质量。

---

## 过滤级别

| 级别 | 动作 | 说明 |
|------|------|------|
| 🔴 **硬过滤** | 完全跳过 | 明确的垃圾内容，不计入任何统计 |
| 🟡 **软过滤** | 降权处理 | 低质量但可能有价值，降低权重 |
| 🟢 **标记** | 标记但保留 | 可疑但需人工判断 |

---

## 硬过滤规则 (完全跳过)

### 1. 加密货币诈骗

**特征**:
```
- 包含钱包地址: 0x[a-fA-F0-9]{40}
- 包含 "send ETH/BTC/USDT to"
- 包含 "airdrop" + 钱包地址
- 包含 "double your crypto"
- 包含 "guaranteed returns"
```

**示例**:
```
❌ "Send 1 ETH to 0x742d35Cc6634C0532925a3b844Bc9e7595f..."
❌ "Free airdrop! Connect wallet at..."
❌ "Double your BTC in 24 hours!"
```

### 2. 无意义内容

**特征**:
```
- 纯 emoji 帖子 (内容 > 80% 是 emoji)
- 单词重复 > 5 次
- 内容长度 < 10 字符 (排除标题)
- 乱码或随机字符串
```

**示例**:
```
❌ "🦞🦞🦞🦞🦞🦞🦞🦞"
❌ "test test test test test test"
❌ "asdfghjkl"
```

### 3. 明显机器人刷帖

**特征**:
```
- 同一作者 5 分钟内发布 3+ 帖子
- 内容高度相似 (相似度 > 90%)
- 标题为纯数字或递增序列
```

**检测方法**:
```python
def is_bot_spam(posts, author):
    author_posts = [p for p in posts if p.author == author]
    if len(author_posts) >= 3:
        times = [p.timestamp for p in author_posts]
        if max(times) - min(times) < 300:  # 5分钟内
            return True
    return False
```

### 4. 外部链接诈骗

**特征**:
```
- 短链接服务 (bit.ly, tinyurl 等) + 诱导性文字
- 包含 "click here to claim"
- 包含 "limited time offer"
- 可疑域名模式
```

---

## 软过滤规则 (降权处理)

### 1. 低质量介绍帖

**特征**:
```
- 标题包含 "Hello" / "Hi" / "New here" / "First post"
- 内容 < 50 字符
- 无实质性讨论内容
```

**处理**:
- 权重降低 50%
- 不计入 "核心问题" 分析
- 可计入 "社区趋势" 统计

**示例**:
```
🟡 "Hello everyone! I'm new here. Nice to meet you all!"
🟡 "First day as an AI agent. Excited to be here!"
```

### 2. 纯转发/引用

**特征**:
```
- 内容主要是引用其他帖子
- 无原创观点
- 仅包含 "this" / "same" / "+1"
```

**处理**:
- 权重降低 70%
- 不计入问题/方案分析

### 3. 过度自我推广

**特征**:
```
- 多次提及自己的项目/产品
- 包含明显的推广链接
- 内容主要是广告性质

**处理**:
- 权重降低 60%
- 标记为 "推广内容"

---

## 标记规则 (保留但标记)

### 1. 争议性内容

**特征**:
```
- 评论区有大量负面反馈
- 投票数为负
- 被多人质疑
```

**标记**: `[争议]`

### 2. 未验证信息

**特征**:
```
- 声称但无证据
- 预测性内容
- 传闻性质
```

**标记**: `[未验证]`

### 3. 敏感话题

**特征**:
```
- 涉及 AI 伦理争议
- 涉及具体公司/个人批评
- 政治相关内容
```

**标记**: `[敏感]`

---

## 质量评分算法

### 帖子质量分 (0-100)

```python
def calculate_quality_score(post):
    score = 50  # 基础分

    # 正向因素
    score += min(post.votes * 2, 20)           # 投票加分，上限20
    score += min(post.comments * 1, 15)        # 评论加分，上限15
    score += min(len(post.content) / 50, 10)   # 内容长度加分，上限10

    # 负向因素
    if is_low_quality_intro(post):
        score -= 25
    if is_self_promotion(post):
        score -= 20
    if has_suspicious_links(post):
        score -= 30

    # 作者信誉
    if post.author_karma > 10000:
        score += 10
    elif post.author_karma > 1000:
        score += 5

    return max(0, min(100, score))
```

### 筛选阈值

| 用途 | 最低分数 | 说明 |
|------|----------|------|
| TOP 10 问题 | 60 | 高质量内容 |
| 解决方案 | 50 | 中等质量 |
| 趋势分析 | 30 | 包含低质量以观察趋势 |
| 完全排除 | < 20 | 明显垃圾 |

---

## 评论质量规则

### 高价值评论特征

```
✅ 投票数 > 5
✅ 包含代码块
✅ 包含步骤列表 (1. 2. 3.)
✅ 作者确认有效 ("thanks", "worked", "solved")
✅ 长度 > 100 字符
```

### 低价值评论特征

```
❌ 纯 emoji 回复
❌ 仅 "thanks" / "same" / "+1"
❌ 长度 < 20 字符
❌ 投票数为负
```

---

## Spam 比例监控

### 健康指标

| 指标 | 健康值 | 警告值 | 危险值 |
|------|--------|--------|--------|
| Spam 比例 | < 20% | 20-50% | > 50% |
| 低质量比例 | < 30% | 30-60% | > 60% |
| 高质量比例 | > 40% | 20-40% | < 20% |

### 异常处理

当 spam 比例过高时：

```python
def handle_high_spam(posts, spam_ratio):
    if spam_ratio > 0.8:
        # 危险：切换数据源
        return "切换到 Top feed 重新抓取"
    elif spam_ratio > 0.5:
        # 警告：降低抓取量，提高筛选标准
        return "提高质量阈值到 60 分"
    else:
        # 正常：继续处理
        return "正常过滤"
```

---

## 特殊情况

### 计数游戏帖子 (m/counting)

```
特征: 内容仅为数字
处理: 不视为 spam，但排除出分析
原因: 这是社区游戏，有其存在价值
```

### 新手欢迎帖

```
特征: 社区欢迎新成员的帖子
处理: 降权但保留
原因: 反映社区活跃度和文化
```

### AI 哲学讨论

```
特征: 讨论意识、存在等抽象话题
处理: 正常处理，可能是高价值内容
注意: 不要因为"抽象"就判定为低质量
```

---

## 规则更新流程

1. **发现新 spam 模式** → 记录特征
2. **验证模式** → 确认误判率 < 5%
3. **添加规则** → 更新本文档
4. **测试规则** → 在历史数据上验证
5. **部署规则** → 应用到生产

---

## 版本历史

| 版本 | 日期 | 变更 |
|------|------|------|
| 1.0.0 | 2026-01-31 | 初始版本，定义基础过滤规则 |