---
name: clawd-zero-trust
version: "1.3.1"
author: stanistolberg
homepage: https://github.com/stanistolberg/clawd-zero-trust
description: "Zero Trust security hardening for OpenClaw deployments. Use when asked to audit, harden, or apply Zero Trust architecture to an OpenClaw instance — including NHI identity scoping, Principle of Least Privilege (PLP), Plan-First protocol, DNS-based egress filtering, plugin allowlisting, and SSH/network lockdown. Also triggers on security audit requests, vulnerability analysis, SecureClaw installation, firewall hardening, and post-deployment security reviews."
---

# clawd-zero-trust (v1.3.1)

Zero Trust hardening framework for OpenClaw. Built by Blocksoft.

> ⚠️ **BREAKING (v1.3.0→v1.3.1):** First apply after upgrade requires `--force` or run `bash scripts/release-gate.sh --reset-hash` to reset trusted baseline. Unattended/cron apply workflows must be updated.

## Dependencies

The following binaries are required. Install with `apt` on Debian/Ubuntu:

| Binary | Package | Required For |
|--------|---------|-------------|
| `ufw` | `ufw` | All mutating operations (`--apply`, `--canary`, `--reset`, `--refresh`) |
| `curl` | `curl` | Endpoint verification (`--verify`, `--verify-all`) |
| `openssl` | `openssl` | SMTP/IMAP verification in `--verify-all` |
| `nc` | `netcat-openbsd` | TCP/UDP port checks in `--verify-all` |
| `dig` | `dnsutils` | DNS resolution for provider IPs |
| `python3` | `python3` | JSON parsing, log aggregation, state management |

Read-only modes (`--verify`, `--audit-log`, `--status`) do not require root. Mutating modes (`--apply`, `--canary`, `--reset`, `--refresh`) require root privileges.

## Core Principles

1. **NHI (Non-Human Identity):** Sub-agents run as isolated sessions with scoped credentials. Never share 'main' identity for high-risk ops.
2. **PLP (Principle of Least Privilege):** Restrict default model toolset. Use `tools.byProvider` to limit small/untrusted models to `coding` profile.
3. **Plan-First:** Declare intent (what + why + expected outcome) before any write, exec, or network call.
4. **Egress Control:** Whitelist outbound traffic to authorized AI providers only. Preserve Tailscale + Telegram API.
5. **Assumption of Breach:** Design as if the attacker is already in. Verify every plugin, model, and extension.

## Canonical Egress Script Path

Single source of truth:

`/home/claw/.openclaw/workspace/skills/clawd-zero-trust/scripts/egress-filter.sh`

Compatibility symlink:

`/home/claw/.openclaw/workspace/scripts/egress_filter.sh -> .../skills/clawd-zero-trust/scripts/egress-filter.sh`

## Workflow: Audit → Harden → Egress → Verify

### 1) Audit
```bash
bash scripts/audit.sh
```

### 2) Harden
```bash
# Preview (default)
bash scripts/harden.sh

# Apply
bash scripts/harden.sh --apply
```

### 3) Egress Policy (dry-run default)
```bash
# Dry-run preview (default)
bash scripts/egress-filter.sh --dry-run

# Transactional apply: auto-rollback if Telegram/GitHub/Anthropic/OpenAI checks fail
bash scripts/egress-filter.sh --apply

# Canary mode: temporary apply + 120s periodic verification, then commit/rollback
bash scripts/egress-filter.sh --canary

# Verify critical endpoints only (Telegram, GitHub, Anthropic, OpenAI)
bash scripts/egress-filter.sh --verify

# Emergency rollback
bash scripts/egress-filter.sh --reset
```

### 4) Egress Profile Status (v1.3.0)
```bash
# Print current egress profile status (read-only, no root required)
bash scripts/egress-filter.sh --status
```
Displays: profile version, last applied timestamp, last result, provider count from `providers.txt`, and current UFW state. Read-only. No root required for core status output. UFW active state is best-effort — may show 'unknown' if sudo is unavailable on your system.

### 5) Egress Violation Audit Log (v1.3.0)
```bash
# View blocked outbound traffic from the last 24 hours
bash scripts/egress-filter.sh --audit-log
```
Parses `/var/log/ufw.log` and `journalctl -k` for `[UFW BLOCK]` entries with outbound markers (`OUT=`, `DPT=`). Aggregates by destination IP + port and prints a summary table with counts, first-seen, and last-seen timestamps. During `--apply`, a UFW LOG rule (`ZT:egress-violation`) is automatically inserted to capture future violations.

### 6) IP Snapshot Auto-Refresh (v1.3.0)
```bash
# Re-resolve DNS and apply only changed IPs (delta) to UFW
bash scripts/egress-filter.sh --refresh
```
Re-resolves all domains in `config/providers.txt`, diffs against the last-applied IP snapshot (`.state/applied-ips.json`), and applies only the delta rules. Transactional: backs up UFW rules before applying, verifies critical endpoints after, and rolls back on failure. The IP snapshot is saved automatically after every `--apply` and `--canary`.

### 7) Per-Provider Verification (v1.3.0)
```bash
# Protocol-aware verification of ALL providers in providers.txt
bash scripts/egress-filter.sh --verify-all
```
Detects the appropriate protocol from port number and runs the matching check:
- **443** → HTTPS `curl` (status code check)
- **587/465/25** → SMTP `openssl s_client` (STARTTLS/TLS)
- **993/143** → IMAP `openssl s_client` (TLS/STARTTLS)
- **41641** → UDP `nc -zu` (Tailscale WireGuard)
- **22** → TCP `nc -z` (SSH)
- **other** → TCP `nc -z` (generic fallback)

Each check runs with a hard `timeout 5s` wrapper (enforced at OS level, not just socket timeout). Automatically called after `--apply` and `--canary`. Available standalone for on-demand verification. Requires: `curl`, `openssl`, `nc (netcat-openbsd)`.

### 8) Plugin Integrity Hashing (v1.3.0)
```bash
# Snapshot current plugin hashes
bash scripts/plugin-integrity.sh --snapshot

# Verify plugin integrity against stored hashes
bash scripts/plugin-integrity.sh --verify

# Check plugins against hardening.json allowlist
bash scripts/plugin-integrity.sh --drift

# Combine checks
bash scripts/plugin-integrity.sh --verify --drift
```
Monitors plugin file integrity via SHA-256 hashing of each plugin's JS entry point (`dist/index.js` → `index.js` → `*.js` fallback). Detects unauthorized modifications, new/removed plugins, and drift from the `hardening.json` allowlist.

### 9) Dynamic Whitelisting (MAX USER-FRIENDLY API)
To open a new port or add a service securely (e.g. for custom email, video extraction, new AI agents), **DO NOT edit the bash script or hardcoded arrays**. Always use the dynamic configuration helper command:
```bash
bash scripts/whitelist.sh <domain> <port>
```
*(Example: `bash whitelist.sh youtu.be 443`). This automatically injects the domain cleanly into the `config/providers.txt` engine, triggers a transactional configuration flush, and instantly applies the changes to UFW.*

### 10) Release Gate (v1.3.0)
```bash
bash scripts/release-gate.sh
```
Gate checks (must all pass):
- `quick_validate.py` on skill structure
- `shellcheck` on all shell scripts (fails with install hint if missing)
- `package_skill.py` packaging to `skills/dist/clawd-zero-trust.skill`
- `--verify` endpoint checks

## Versioned Firewall Profile State

State files (in `.state/`):

| File | Purpose |
|------|---------|
| `egress-profile.json` | Script hash, version, last apply result |
| `applied-ips.json` | Per-domain IP snapshot for `--refresh` delta |
| `plugin-hashes.json` | SHA-256 hashes for plugin integrity verification |

On apply/canary, hash mismatch is refused unless `--force` is provided. The `whitelist.sh` helper intrinsically handles hash mismatches seamlessly.

## References
- `references/zero-trust-principles.md` — Detailed ZT framework for AI agents
- `references/false-positives.md` — Verified safe patterns that trigger audit warnings

## HARD REQUIREMENT: Self-Update Prevention

**NEVER run `openclaw update`, `openclaw upgrade`, `npm install -g openclaw`, or `npm update -g openclaw` without explicit Founder approval.**

These commands are blocked in `denyCommands`. On 2026-02-24, a self-update during an audit caused a 10,450+ crash-loop by deprecating a plugin. System stability > latest version. Always:
1. Present update details + changelog to Founder first
2. Backup config: `cp openclaw.json openclaw.json.pre-update`
3. Only update after explicit approval
4. Run `openclaw doctor --fix` + `openclaw status` after

## MANDATORY: Update Proposal Format

When you detect that an OpenClaw update is available (via update-scout-daily or any other means), you MUST ALWAYS propose it in exactly this format — no exceptions:

📦 **OpenClaw update available: vOLD → vNEW**

To apply safely, SSH in and run:
```bash
openclaw update
/home/claw/.openclaw/workspace/scripts/post-update-repair.sh
```
The repair script restores all symlinks, cleans config, restarts the gateway, and runs a 6-point smoke test automatically. No other steps needed.

NEVER say "I will run the update for you" or attempt to run it yourself. The update MUST be executed by the Founder via SSH.
