import * as cborg from 'cborg';
import { MessageType } from '../types.js';

// Length-prefixed framing for wire protocol
// Format: [4-byte big-endian length][message bytes]

const MAX_MESSAGE_SIZE = 16 * 1024 * 1024; // 16 MB

export function frameMessage(data: Uint8Array): Uint8Array {
  const length = data.length;
  if (length > MAX_MESSAGE_SIZE) {
    throw new Error(`Message too large: ${length} > ${MAX_MESSAGE_SIZE}`);
  }

  const framed = new Uint8Array(4 + length);
  const view = new DataView(framed.buffer);
  view.setUint32(0, length, false); // big-endian
  framed.set(data, 4);

  return framed;
}

export function unframeMessage(framed: Uint8Array): { data: Uint8Array; remaining: Uint8Array } | null {
  if (framed.length < 4) {
    return null; // need more data
  }

  const view = new DataView(framed.buffer, framed.byteOffset, framed.length);
  const length = view.getUint32(0, false);

  if (length > MAX_MESSAGE_SIZE) {
    throw new Error(`Invalid message length: ${length}`);
  }

  if (framed.length < 4 + length) {
    return null; // need more data
  }

  return {
    data: framed.slice(4, 4 + length),
    remaining: framed.slice(4 + length),
  };
}

// CBOR encoding for control messages
export interface WireMessage {
  type: MessageType;
  [key: string]: unknown;
}

export function encodeMessage(msg: WireMessage): Uint8Array {
  return cborg.encode(msg);
}

export function decodeMessage(data: Uint8Array): WireMessage {
  return cborg.decode(data) as WireMessage;
}

// Specific message constructors
export function createHelloMessage(nodePublicKey: Uint8Array, nonce: Uint8Array): WireMessage {
  return {
    type: MessageType.HELLO,
    version: 1,
    nodePublicKey,
    nonce,
    timestamp: Math.floor(Date.now() / 1000),
  };
}

export function createAuthMessage(
  principal: string,
  attestation: unknown,
  sessionNonce: Uint8Array
): WireMessage {
  return {
    type: MessageType.AUTH,
    principal,
    attestation,
    sessionNonce,
  };
}

export function createAuthOkMessage(): WireMessage {
  return { type: MessageType.AUTH_OK };
}

export function createAuthFailMessage(code: number, reason: string): WireMessage {
  return {
    type: MessageType.AUTH_FAIL,
    code,
    reason,
  };
}

export function createStreamDataMessage(streamId: bigint, data: Uint8Array, fin = false): WireMessage {
  return {
    type: MessageType.STREAM_DATA,
    streamId,
    data,
    fin,
  };
}

export function createPingMessage(nonce: Uint8Array): WireMessage {
  return {
    type: MessageType.PING,
    nonce,
  };
}

export function createPongMessage(nonce: Uint8Array): WireMessage {
  return {
    type: MessageType.PONG,
    nonce,
  };
}

// Chat protocol messages (application layer, on top of streams)
export interface ChatMessage {
  id: string;
  from: string;
  /** Optional nickname of sender */
  nick?: string;
  content: string;
  timestamp: number;
}

export function encodeChatMessage(msg: ChatMessage): Uint8Array {
  return cborg.encode(msg);
}

export function decodeChatMessage(data: Uint8Array): ChatMessage {
  return cborg.decode(data) as ChatMessage;
}
