// PX-1 Peer Exchange Protocol Types
// Protocol: /snap2p/px1/1.0.0

/**
 * PX-1 Message Types
 */
export enum PX1MessageType {
  /** Request to resolve a principal's multiaddrs */
  PX_RESOLVE_REQ = 0x01,
  /** Response with resolved multiaddrs */
  PX_RESOLVE_RESP = 0x02,
  /** Push peer information proactively */
  PX_PUSH = 0x03,
}

/**
 * Visibility modes for peer sharing
 */
export enum PeerVisibility {
  /** Share with anyone who asks */
  PUBLIC = 'public',
  /** Only share with allowlisted peers */
  PRIVATE = 'private',
  /** Never share, don't respond to resolve requests */
  STEALTH = 'stealth',
}

/**
 * Peer address info for exchange
 */
export interface PeerAddressInfo {
  /** Principal identifier (stacks:<address>) */
  principal: string;
  /** libp2p multiaddrs as strings */
  multiaddrs: string[];
  /** When this info was last updated (Unix ms) */
  lastSeen: number;
  /** libp2p peer ID if known */
  peerId?: string;
}

/**
 * Request to resolve a principal's addresses
 */
export interface PXResolveRequest {
  type: PX1MessageType.PX_RESOLVE_REQ;
  /** Principal to resolve */
  principal: string;
  /** Request ID for correlation */
  requestId: string;
}

/**
 * Response with resolved addresses
 */
export interface PXResolveResponse {
  type: PX1MessageType.PX_RESOLVE_RESP;
  /** Request ID from the request */
  requestId: string;
  /** Found peer info, null if not found or not allowed */
  peer: PeerAddressInfo | null;
  /** Error message if resolution failed */
  error?: string;
}

/**
 * Push peer information to connected peers
 */
export interface PXPush {
  type: PX1MessageType.PX_PUSH;
  /** Peers being shared */
  peers: PeerAddressInfo[];
}

/**
 * Union type for all PX-1 messages
 */
export type PX1Message = PXResolveRequest | PXResolveResponse | PXPush;

/**
 * Local cache entry for peer addresses
 */
export interface PeerCacheEntry extends PeerAddressInfo {
  /** Which peer told us about this (for provenance) */
  source: string;
  /** When we cached this */
  cachedAt: number;
  /** Whether we've directly verified this peer */
  verified: boolean;
}

/**
 * PX-1 configuration
 */
export interface PX1Config {
  /** Our visibility mode */
  visibility: PeerVisibility;
  /** Principals allowed to see us in private mode */
  allowlist: string[];
  /** Maximum peers to push at once */
  maxPushPeers: number;
  /** Cache TTL in milliseconds */
  cacheTTL: number;
}

/**
 * Default PX-1 configuration
 */
export const DEFAULT_PX1_CONFIG: PX1Config = {
  visibility: PeerVisibility.PUBLIC,
  allowlist: [],
  maxPushPeers: 20,
  cacheTTL: 5 * 60 * 1000, // 5 minutes
};
