/**
 * Gateway Architecture Type Definitions
 *
 * Phase 1: Core Infrastructure
 * These types support the multi-identity gateway architecture.
 */

import type { FullIdentity } from '../identity/keys.js';
import type { Message, Peer } from '../types.js';

/**
 * Configuration for a single identity in gateway mode
 */
export interface IdentityConfig {
  /** Stacks principal (e.g., "stacks:ST1ABC...") */
  principal: string;

  /** Optional nickname for easier reference */
  nick?: string;

  /** Whether to load this identity automatically on daemon start */
  autoload: boolean;

  /** Whether to allow local IPC connections to this identity */
  allowLocal: boolean;

  /**
   * Remote peers allowed to send messages to this identity
   * Use ["*"] for all peers, or list specific principals
   */
  allowedRemotePeers: string[];

  /** Whether to enable OpenClaw wake for this identity */
  openclawWake: boolean;
}

/**
 * Gateway configuration file structure
 * Stored in ~/.clawchat/gateway-config.json
 */
export interface GatewayConfig {
  /** Configuration version (currently 1) */
  version: number;

  /** P2P port for the gateway daemon */
  p2pPort: number;

  /** List of identities managed by this gateway */
  identities: IdentityConfig[];
}

/**
 * Runtime representation of a loaded identity
 * Managed by IdentityManager
 */
export interface LoadedIdentity {
  /** The decrypted identity */
  identity: FullIdentity;

  /** Configuration for this identity */
  config: IdentityConfig;

  /** Per-identity inbox messages */
  inbox: Message[];

  /** Per-identity outbox messages */
  outbox: Message[];

  /** Per-identity peer list */
  peers: Peer[];

  /** Data directory for this identity (identities/{principal}/) */
  dataDir: string;
}

/**
 * Validation error for gateway configuration
 */
export class GatewayConfigError extends Error {
  constructor(message: string) {
    super(message);
    this.name = 'GatewayConfigError';
  }
}
