/**
 * SNaP2P Protocol Handler for libp2p
 *
 * Protocol: /snap2p/1.0.0
 *
 * Handles the authentication exchange over libp2p streams:
 * 1. HELLO exchange (node public keys)
 * 2. AUTH exchange (Stacks attestations)
 * 3. AUTH_OK confirmation
 *
 * After authentication, the stream can be used for application messages.
 */

import { EventEmitter } from 'events';
import * as cborg from 'cborg';
import { randomBytes } from '@noble/hashes/utils';
import { pipe } from 'it-pipe';
import { pushable, type Pushable } from 'it-pushable';
import type { Libp2p } from 'libp2p';
import type { Stream } from '@libp2p/interface';
import { MessageType, type NodeKeyAttestation } from '../types.js';
import {
  createAttestation,
  verifyAttestation,
  type FullIdentity,
} from '../identity/keys.js';
import {
  frameMessage,
  unframeMessage,
  encodeMessage,
  decodeMessage,
  createHelloMessage,
  createAuthMessage,
  createAuthOkMessage,
  createPongMessage,
  encodeChatMessage,
  decodeChatMessage,
  type WireMessage,
  type ChatMessage,
} from '../wire/framing.js';
import { SNAP2P_PROTOCOL } from './libp2p-node.js';

/**
 * Authenticated session over libp2p stream
 */
export class Snap2pSession extends EventEmitter {
  private stream: Stream;
  private identity: FullIdentity;
  private remotePrincipal: string | null = null;
  private remotePeerId: string;
  private buffer: Uint8Array = new Uint8Array(0);
  private authenticated = false;
  private closed = false;
  private outgoing: Pushable<Uint8Array>;
  private messageQueue: WireMessage[] = [];
  private messageResolvers: Array<(msg: WireMessage) => void> = [];

  constructor(stream: Stream, identity: FullIdentity, remotePeerId: string) {
    super();
    this.stream = stream;
    this.identity = identity;
    this.remotePeerId = remotePeerId;
    this.outgoing = pushable<Uint8Array>();

    // Set up the stream pipeline
    this.setupPipeline();
  }

  private setupPipeline(): void {
    // Pipe outgoing messages to the stream sink
    pipe(
      this.outgoing,
      this.stream.sink
    ).catch((err) => {
      if (!this.closed) {
        this.emit('error', err);
      }
    });

    // Read incoming messages from the stream source
    this.readLoop();
  }

  private async readLoop(): Promise<void> {
    try {
      for await (const chunk of this.stream.source) {
        if (this.closed) break;

        // Convert BufferList or Uint8Array to Uint8Array
        let data: Uint8Array;
        if (chunk instanceof Uint8Array) {
          data = chunk;
        } else if (typeof chunk.subarray === 'function') {
          // Uint8ArrayList has a subarray method
          data = Uint8Array.from(chunk.subarray());
        } else {
          // Fallback for other iterable types
          data = Uint8Array.from(chunk as Iterable<number>);
        }

        // Append to buffer
        const newBuffer = new Uint8Array(this.buffer.length + data.length);
        newBuffer.set(this.buffer);
        newBuffer.set(data, this.buffer.length);
        this.buffer = newBuffer;

        // Process complete messages
        while (true) {
          const result = unframeMessage(this.buffer);
          if (!result) break;

          this.buffer = result.remaining;
          const msg = decodeMessage(result.data);
          this.handleIncomingMessage(msg);
        }
      }
    } catch (err) {
      if (!this.closed) {
        this.emit('error', err);
      }
    } finally {
      if (!this.closed) {
        this.emit('close');
      }
    }
  }

  private handleIncomingMessage(msg: WireMessage): void {
    // Check if anyone is waiting for a message
    if (this.messageResolvers.length > 0) {
      const resolver = this.messageResolvers.shift()!;
      resolver(msg);
      return;
    }

    // Queue the message for later
    this.messageQueue.push(msg);
  }

  private async waitForMessage(): Promise<WireMessage> {
    // Check queue first
    if (this.messageQueue.length > 0) {
      return this.messageQueue.shift()!;
    }

    // Wait for a message
    return new Promise((resolve) => {
      this.messageResolvers.push(resolve);
    });
  }

  get isAuthenticated(): boolean {
    return this.authenticated;
  }

  get remote(): string | null {
    return this.remotePrincipal;
  }

  get peerId(): string {
    return this.remotePeerId;
  }

  /**
   * Get the local identity principal for this session
   */
  get localPrincipal(): string {
    return this.identity.principal;
  }

  /**
   * Perform authentication as initiator (we opened the stream)
   */
  async authenticateAsInitiator(): Promise<void> {
    const nonce = randomBytes(16);
    const attestation = await createAttestation(this.identity);

    // Send HELLO
    this.sendMessage(createHelloMessage(this.identity.publicKey, nonce));

    // Send AUTH
    this.sendMessage(createAuthMessage(
      this.identity.principal,
      this.serializeAttestation(attestation),
      nonce
    ));

    // Receive their HELLO
    const theirHello = await this.waitForMessage();
    if (theirHello.type !== MessageType.HELLO) {
      throw new Error(`Expected HELLO, got ${theirHello.type}`);
    }

    // Receive their AUTH
    const theirAuth = await this.waitForMessage();
    if (theirAuth.type !== MessageType.AUTH) {
      throw new Error(`Expected AUTH, got ${theirAuth.type}`);
    }

    // Verify attestation
    const theirAttestation = this.parseAttestation(theirAuth.attestation);
    if (!verifyAttestation(theirAttestation, this.identity.testnet)) {
      throw new Error('Invalid attestation');
    }

    this.remotePrincipal = theirAuth.principal as string;

    // Send AUTH_OK
    this.sendMessage(createAuthOkMessage());

    // Receive AUTH_OK
    const authOk = await this.waitForMessage();
    if (authOk.type !== MessageType.AUTH_OK) {
      throw new Error('Authentication failed');
    }

    this.authenticated = true;
    this.emit('authenticated', this.remotePrincipal);

    // Start processing application messages
    this.startMessageLoop();
  }

  /**
   * Perform authentication as responder (they opened the stream)
   */
  async authenticateAsResponder(): Promise<void> {
    // Receive their HELLO
    const theirHello = await this.waitForMessage();
    if (theirHello.type !== MessageType.HELLO) {
      throw new Error(`Expected HELLO, got ${theirHello.type}`);
    }

    // Receive their AUTH
    const theirAuth = await this.waitForMessage();
    if (theirAuth.type !== MessageType.AUTH) {
      throw new Error(`Expected AUTH, got ${theirAuth.type}`);
    }

    // Verify attestation
    const theirAttestation = this.parseAttestation(theirAuth.attestation);
    if (!verifyAttestation(theirAttestation, this.identity.testnet)) {
      throw new Error('Invalid attestation');
    }

    this.remotePrincipal = theirAuth.principal as string;

    // Send our HELLO
    const nonce = randomBytes(16);
    const attestation = await createAttestation(this.identity);
    this.sendMessage(createHelloMessage(this.identity.publicKey, nonce));

    // Send our AUTH
    this.sendMessage(createAuthMessage(
      this.identity.principal,
      this.serializeAttestation(attestation),
      nonce
    ));

    // Receive AUTH_OK
    const authOk = await this.waitForMessage();
    if (authOk.type !== MessageType.AUTH_OK) {
      throw new Error('Authentication failed');
    }

    // Send AUTH_OK
    this.sendMessage(createAuthOkMessage());

    this.authenticated = true;
    this.emit('authenticated', this.remotePrincipal);

    // Start processing application messages
    this.startMessageLoop();
  }

  private serializeAttestation(attestation: NodeKeyAttestation): Record<string, unknown> {
    return {
      version: attestation.version,
      principal: attestation.principal,
      nodePublicKey: Array.from(attestation.nodePublicKey),
      issuedAt: attestation.issuedAt,
      expiresAt: attestation.expiresAt,
      nonce: Array.from(attestation.nonce),
      domain: attestation.domain,
      signature: Array.from(attestation.signature),
    };
  }

  private parseAttestation(attestation: unknown): NodeKeyAttestation {
    const att = attestation as {
      version: number;
      principal: string;
      nodePublicKey: number[] | Uint8Array;
      issuedAt: number;
      expiresAt: number;
      nonce: number[] | Uint8Array;
      domain: string;
      signature: number[] | Uint8Array;
    };

    return {
      version: att.version,
      principal: att.principal,
      nodePublicKey: new Uint8Array(att.nodePublicKey),
      issuedAt: att.issuedAt,
      expiresAt: att.expiresAt,
      nonce: new Uint8Array(att.nonce),
      domain: att.domain,
      signature: new Uint8Array(att.signature),
    };
  }

  private startMessageLoop(): void {
    const processMessages = async () => {
      while (!this.closed && this.authenticated) {
        try {
          const msg = await this.waitForMessage();

          if (msg.type === MessageType.STREAM_DATA) {
            const chatMsg = decodeChatMessage(msg.data as Uint8Array);
            this.emit('message', chatMsg);
          } else if (msg.type === MessageType.PING) {
            this.sendMessage(createPongMessage(msg.nonce as Uint8Array));
          }
        } catch (err) {
          if (!this.closed) {
            this.emit('error', err);
          }
          break;
        }
      }
    };

    processMessages();
  }

  private sendMessage(msg: WireMessage): void {
    const encoded = encodeMessage(msg);
    const framed = frameMessage(encoded);
    this.outgoing.push(framed);
  }

  /**
   * Send a chat message
   */
  async sendChatMessage(content: string): Promise<void> {
    if (!this.authenticated) {
      throw new Error('Not authenticated');
    }

    const msg: ChatMessage = {
      id: Buffer.from(randomBytes(16)).toString('hex'),
      from: this.identity.principal,
      nick: this.identity.nick,
      content,
      timestamp: Date.now(),
    };

    const encoded = encodeChatMessage(msg);
    this.sendMessage({
      type: MessageType.STREAM_DATA,
      streamId: 0n,
      data: encoded,
      fin: false,
    });
  }

  /**
   * Close the session
   */
  close(): void {
    this.closed = true;
    this.outgoing.end();
    try {
      this.stream.close();
    } catch {
      // Ignore close errors
    }
  }
}

/**
 * SNaP2P protocol handler for libp2p
 */
/**
 * Callback to determine which identity to use for a session
 * Used in gateway mode to select identity based on remote peer
 */
export type IdentityResolver = (remotePrincipal?: string) => FullIdentity | null;

export class Snap2pProtocolHandler extends EventEmitter {
  private node: Libp2p;
  private identityResolver: IdentityResolver;
  private sessions: Map<string, Snap2pSession> = new Map();

  /**
   * Create a new SNaP2P protocol handler
   * @param node - libp2p node
   * @param identityResolver - Function to resolve which identity to use
   */
  constructor(node: Libp2p, identityResolver: IdentityResolver) {
    super();
    this.node = node;
    this.identityResolver = identityResolver;
  }

  /**
   * Register the protocol handler
   */
  start(): void {
    this.node.handle(SNAP2P_PROTOCOL, async ({ stream, connection }) => {
      const remotePeerId = connection.remotePeer.toString();

      try {
        // Use resolver to get default identity for initial auth
        const resolvedIdentity = this.identityResolver();
        if (!resolvedIdentity) {
          console.error('[snap2p] No identity available for incoming connection');
          stream.close();
          return;
        }

        const session = new Snap2pSession(stream, resolvedIdentity, remotePeerId);

        session.on('authenticated', (principal) => {
          // Verify the remote peer is allowed
          const targetIdentity = this.identityResolver(principal);
          if (!targetIdentity) {
            console.warn(`[snap2p] No identity allows peer ${principal}, closing connection`);
            session.close();
            return;
          }

          this.sessions.set(principal, session);
          this.emit('session', session);
        });

        session.on('message', (msg) => {
          this.emit('message', msg, session);
        });

        session.on('close', () => {
          if (session.remote) {
            this.sessions.delete(session.remote);
          }
          this.emit('session:close', session);
        });

        session.on('error', (err) => {
          this.emit('session:error', err, session);
        });

        await session.authenticateAsResponder();
      } catch (err) {
        console.error('SNaP2P auth failed:', err);
        stream.close();
      }
    });
  }

  /**
   * Connect to a peer and authenticate
   * @param peerId - libp2p peer ID to connect to
   * @param identity - Optional: specific identity to use, otherwise uses default from resolver
   */
  async connect(peerId: string, identity?: FullIdentity): Promise<Snap2pSession> {
    // Open a stream to the peer
    const { peerIdFromString } = await import('@libp2p/peer-id');
    const remotePeer = peerIdFromString(peerId);

    const stream = await this.node.dialProtocol(remotePeer, SNAP2P_PROTOCOL);

    // Determine which identity to use
    let sessionIdentity: FullIdentity;
    if (identity) {
      // Explicit identity provided
      sessionIdentity = identity;
    } else {
      // Use resolver to get default identity
      const resolvedIdentity = this.identityResolver();
      if (!resolvedIdentity) {
        throw new Error('No identity available for outgoing connection');
      }
      sessionIdentity = resolvedIdentity;
    }

    const session = new Snap2pSession(stream, sessionIdentity, peerId);

    session.on('authenticated', (principal) => {
      this.sessions.set(principal, session);
      this.emit('session', session);
    });

    session.on('message', (msg) => {
      this.emit('message', msg, session);
    });

    session.on('close', () => {
      if (session.remote) {
        this.sessions.delete(session.remote);
      }
      this.emit('session:close', session);
    });

    session.on('error', (err) => {
      this.emit('session:error', err, session);
    });

    await session.authenticateAsInitiator();

    return session;
  }

  /**
   * Get a session by principal
   */
  getSession(principal: string): Snap2pSession | undefined {
    return this.sessions.get(principal);
  }

  /**
   * Get all active sessions
   */
  getSessions(): Snap2pSession[] {
    return Array.from(this.sessions.values());
  }

  /**
   * Stop the protocol handler
   */
  stop(): void {
    this.node.unhandle(SNAP2P_PROTOCOL);
    for (const session of this.sessions.values()) {
      session.close();
    }
    this.sessions.clear();
  }
}
