/**
 * Session management for clawchat
 *
 * This module provides two session implementations:
 * 1. LegacySession - Raw TCP with custom Noise handshake (original)
 * 2. Snap2pSession - libp2p-based with NAT traversal (re-exported)
 *
 * The new libp2p-based Snap2pSession is recommended for production use
 * as it provides NAT traversal and mesh networking capabilities.
 */

import * as net from 'net';
import { EventEmitter } from 'events';
import {
  initHandshake,
  initiatorMsg1,
  initiatorMsg3,
  responderMsg2,
  responderFinalize,
  deriveSessionKeys,
  encrypt,
  decrypt,
  generateX25519KeyPair,
  type SessionKeys,
  type KeyPair,
} from '../crypto/noise.js';
import {
  frameMessage,
  unframeMessage,
  encodeMessage,
  decodeMessage,
  createHelloMessage,
  createAuthMessage,
  createAuthOkMessage,
  createPongMessage,
  encodeChatMessage,
  decodeChatMessage,
  type WireMessage,
  type ChatMessage,
} from '../wire/framing.js';
import { MessageType, type NodeKeyAttestation } from '../types.js';
import { createAttestation, verifyAttestation, type FullIdentity } from '../identity/keys.js';
import { randomBytes } from '@noble/hashes/utils';

// Re-export the new libp2p-based session
export { Snap2pSession, Snap2pProtocolHandler } from './snap2p-protocol.js';

export interface SessionConfig {
  identity: FullIdentity;
  socket: net.Socket;
  initiator: boolean;
}

/**
 * Legacy TCP session with custom Noise handshake
 *
 * @deprecated Use Snap2pSession from './snap2p-protocol.js' for new code.
 * This class is kept for backwards compatibility and testing.
 */
export class Session extends EventEmitter {
  private socket: net.Socket;
  private identity: FullIdentity;
  private initiator: boolean;
  private sessionKeys: SessionKeys | null = null;
  private buffer: Uint8Array = new Uint8Array(0);
  private remotePrincipal: string | null = null;
  private x25519KeyPair: KeyPair;

  constructor(config: SessionConfig) {
    super();
    this.socket = config.socket;
    this.identity = config.identity;
    this.initiator = config.initiator;

    // Generate X25519 keypair for this session
    this.x25519KeyPair = generateX25519KeyPair();

    this.socket.on('data', (data) => this.onData(data));
    this.socket.on('error', (err) => this.emit('error', err));
    this.socket.on('close', () => this.emit('close'));
  }

  get isAuthenticated(): boolean {
    return this.sessionKeys !== null && this.remotePrincipal !== null;
  }

  get remote(): string | null {
    return this.remotePrincipal;
  }

  async handshake(): Promise<void> {
    if (this.initiator) {
      await this.initiatorHandshake();
    } else {
      await this.responderHandshake();
    }
  }

  private async initiatorHandshake(): Promise<void> {
    const state = initHandshake(this.x25519KeyPair, true);

    // -> e (32 bytes raw)
    const msg1 = initiatorMsg1(state);
    this.socket.write(Buffer.from(msg1));

    // <- e, ee, s, es (64 bytes raw)
    const msg2 = await this.readExact(64);
    const msg3 = initiatorMsg3(state, msg2);

    // -> s, se (32 bytes raw)
    this.socket.write(Buffer.from(msg3));

    // Derive session keys
    this.sessionKeys = deriveSessionKeys(state, true);

    // Exchange HELLO + AUTH (now framed and encrypted)
    await this.exchangeAuth();
  }

  private async responderHandshake(): Promise<void> {
    const state = initHandshake(this.x25519KeyPair, false);

    // <- e (32 bytes raw)
    const msg1 = await this.readExact(32);

    // -> e, ee, s, es (64 bytes raw)
    const msg2 = responderMsg2(state, msg1);
    this.socket.write(Buffer.from(msg2));

    // <- s, se (32 bytes raw)
    const msg3 = await this.readExact(32);
    responderFinalize(state, msg3);

    // Derive session keys
    this.sessionKeys = deriveSessionKeys(state, false);

    // Exchange HELLO + AUTH (now framed and encrypted)
    await this.exchangeAuth();
  }

  private async exchangeAuth(): Promise<void> {
    if (!this.sessionKeys) throw new Error('No session keys');

    const nonce = randomBytes(16);
    const attestation = await createAttestation(this.identity);

    // Send HELLO
    const hello = createHelloMessage(this.identity.publicKey, nonce);
    this.sendEncrypted(hello);

    // Send AUTH
    const auth = createAuthMessage(this.identity.principal, attestation, nonce);
    this.sendEncrypted(auth);

    // Receive their HELLO + AUTH
    const theirHello = await this.recvEncrypted();
    if (theirHello.type !== MessageType.HELLO) {
      throw new Error('Expected HELLO message');
    }

    const theirAuth = await this.recvEncrypted();
    if (theirAuth.type !== MessageType.AUTH) {
      throw new Error('Expected AUTH message');
    }

    // Verify attestation
    const theirAttestation = theirAuth.attestation as NodeKeyAttestation;
    if (!verifyAttestation(theirAttestation, this.identity.testnet)) {
      throw new Error('Invalid attestation');
    }

    this.remotePrincipal = theirAuth.principal as string;

    // Send AUTH_OK
    this.sendEncrypted(createAuthOkMessage());

    // Receive AUTH_OK
    const authOk = await this.recvEncrypted();
    if (authOk.type !== MessageType.AUTH_OK) {
      throw new Error('Authentication failed');
    }

    this.emit('authenticated', this.remotePrincipal);

    // Start message read loop
    this.startReadLoop();
  }

  private startReadLoop(): void {
    const readLoop = async () => {
      while (this.sessionKeys && this.socket.readable) {
        try {
          const msg = await this.recvEncrypted();
          if (msg.type === MessageType.STREAM_DATA) {
            const chatMsg = decodeChatMessage(msg.data as Uint8Array);
            this.emit('message', chatMsg);
          } else if (msg.type === MessageType.PING) {
            this.sendEncrypted(createPongMessage(msg.nonce as Uint8Array));
          }
        } catch {
          break; // Connection closed or error
        }
      }
    };
    readLoop();
  }

  private sendEncrypted(msg: WireMessage): void {
    if (!this.sessionKeys) throw new Error('No session keys');

    const encoded = encodeMessage(msg);
    const encrypted = encrypt(this.sessionKeys.sendCipher, encoded);
    this.socket.write(frameMessage(encrypted));
  }

  private async recvEncrypted(): Promise<WireMessage> {
    if (!this.sessionKeys) throw new Error('No session keys');

    const encrypted = await this.readFrame();
    const decrypted = decrypt(this.sessionKeys.recvCipher, encrypted);
    return decodeMessage(decrypted);
  }

  // Read exactly n bytes (for raw handshake messages)
  private readExact(n: number): Promise<Uint8Array> {
    return new Promise((resolve, reject) => {
      let resolved = false;

      const tryRead = () => {
        if (resolved) return;
        if (this.buffer.length >= n) {
          const data = this.buffer.slice(0, n);
          this.buffer = this.buffer.slice(n);
          resolved = true;
          cleanup();
          resolve(data);
          return true;
        }
        // Schedule another check
        setImmediate(tryRead);
        return false;
      };

      const cleanup = () => {
        this.socket.off('error', onError);
        this.socket.off('close', onClose);
      };

      const onError = (err: Error) => {
        if (resolved) return;
        resolved = true;
        cleanup();
        reject(err);
      };

      const onClose = () => {
        if (resolved) return;
        resolved = true;
        cleanup();
        reject(new Error('Connection closed'));
      };

      this.socket.on('error', onError);
      this.socket.on('close', onClose);

      // Start polling the buffer
      tryRead();
    });
  }

  // Read a length-prefixed frame (for encrypted messages)
  private readFrame(): Promise<Uint8Array> {
    return new Promise((resolve, reject) => {
      let resolved = false;

      const tryRead = () => {
        if (resolved) return;
        const result = unframeMessage(this.buffer);
        if (result) {
          this.buffer = result.remaining;
          resolved = true;
          cleanup();
          resolve(result.data);
          return true;
        }
        // Schedule another check
        setImmediate(tryRead);
        return false;
      };

      const cleanup = () => {
        this.socket.off('error', onError);
        this.socket.off('close', onClose);
      };

      const onError = (err: Error) => {
        if (resolved) return;
        resolved = true;
        cleanup();
        reject(err);
      };

      const onClose = () => {
        if (resolved) return;
        resolved = true;
        cleanup();
        reject(new Error('Connection closed'));
      };

      this.socket.on('error', onError);
      this.socket.on('close', onClose);

      // Start polling the buffer
      tryRead();
    });
  }

  private onData(data: Buffer): void {
    // Buffer incoming data for readExact/readFrame to consume
    const newBuffer = new Uint8Array(this.buffer.length + data.length);
    newBuffer.set(this.buffer);
    newBuffer.set(data, this.buffer.length);
    this.buffer = newBuffer;
  }

  // Send a chat message
  sendMessage(content: string): void {
    if (!this.sessionKeys) throw new Error('Not authenticated');

    const msg: ChatMessage = {
      id: Buffer.from(randomBytes(16)).toString('hex'),
      from: this.identity.principal,
      nick: this.identity.nick,
      content,
      timestamp: Date.now(),
    };

    const encoded = encodeChatMessage(msg);
    const streamData = {
      type: MessageType.STREAM_DATA,
      streamId: 0n,
      data: encoded,
      fin: false,
    };

    this.sendEncrypted(streamData);
  }

  // Receive messages (call in a loop or on data event)
  async receiveMessage(): Promise<ChatMessage | null> {
    try {
      const msg = await this.recvEncrypted();
      if (msg.type === MessageType.STREAM_DATA) {
        const chatMsg = decodeChatMessage(msg.data as Uint8Array);
        this.emit('message', chatMsg);
        return chatMsg;
      }
      return null;
    } catch {
      return null;
    }
  }

  close(): void {
    this.socket.end();
  }
}

/**
 * Connect to a peer using legacy TCP
 *
 * @deprecated Use libp2p dialPeer and Snap2pProtocolHandler for new code.
 */
export async function connect(
  address: string,
  identity: FullIdentity
): Promise<Session> {
  const [host, portStr] = address.split(':');
  const port = parseInt(portStr, 10);

  return new Promise((resolve, reject) => {
    const socket = net.createConnection({ host, port }, async () => {
      try {
        const session = new Session({ identity, socket, initiator: true });
        await session.handshake();
        resolve(session);
      } catch (err) {
        socket.destroy();
        reject(err);
      }
    });

    socket.on('error', reject);
  });
}

/**
 * Listen for legacy TCP connections
 *
 * @deprecated Use libp2p node with Snap2pProtocolHandler for new code.
 */
export function listen(
  port: number,
  identity: FullIdentity,
  onSession: (session: Session) => void
): net.Server {
  const server = net.createServer(async (socket) => {
    try {
      const session = new Session({ identity, socket, initiator: false });
      await session.handshake();
      onSession(session);
    } catch (err) {
      console.error('Handshake failed:', err);
      socket.destroy();
    }
  });

  server.listen(port);
  return server;
}
