/**
 * PX-1 Peer Exchange Protocol
 *
 * Protocol: /snap2p/px1/1.0.0
 *
 * Enables mesh connectivity by allowing peers to:
 * 1. Push known peer addresses (PX_PUSH)
 * 2. Resolve peer addresses by principal (PX_RESOLVE_REQ/RESP)
 *
 * Respects visibility modes:
 * - PUBLIC: Share with anyone
 * - PRIVATE: Only share with allowlisted peers
 * - STEALTH: Never share
 */

import { EventEmitter } from 'events';
import * as cborg from 'cborg';
import { randomBytes } from '@noble/hashes/utils';
import { pipe } from 'it-pipe';
import { pushable } from 'it-pushable';
import type { Libp2p } from 'libp2p';
import type { Stream } from '@libp2p/interface';
import {
  PX1MessageType,
  PeerVisibility,
  type PX1Message,
  type PXResolveRequest,
  type PXResolveResponse,
  type PXPush,
  type PeerAddressInfo,
  type PeerCacheEntry,
  type PX1Config,
  DEFAULT_PX1_CONFIG,
} from '../types/px1.js';
import { PX1_PROTOCOL } from './libp2p-node.js';
import { frameMessage, unframeMessage } from '../wire/framing.js';

/**
 * PX-1 Peer Exchange Handler
 */
export class PX1Handler extends EventEmitter {
  private node: Libp2p;
  private config: PX1Config;
  private peerCache: Map<string, PeerCacheEntry> = new Map();
  private ourPrincipal: string;
  private ourPeerId: string;

  constructor(
    node: Libp2p,
    principal: string,
    config: Partial<PX1Config> = {}
  ) {
    super();
    this.node = node;
    this.ourPrincipal = principal;
    this.ourPeerId = node.peerId.toString();
    this.config = { ...DEFAULT_PX1_CONFIG, ...config };
  }

  /**
   * Register the PX-1 protocol handler
   */
  start(): void {
    this.node.handle(PX1_PROTOCOL, async ({ stream, connection }) => {
      try {
        await this.handleIncomingStream(stream, connection.remotePeer.toString());
      } catch (err) {
        console.error('PX-1 handler error:', err);
        stream.close();
      }
    });

    // Periodically clean up expired cache entries
    setInterval(() => this.cleanupCache(), 60000);
  }

  /**
   * Stop the PX-1 protocol handler
   */
  stop(): void {
    this.node.unhandle(PX1_PROTOCOL);
    this.peerCache.clear();
  }

  /**
   * Handle incoming PX-1 stream
   */
  private async handleIncomingStream(stream: Stream, remotePeerId: string): Promise<void> {
    let buffer = new Uint8Array(0);

    // Read messages from stream
    for await (const chunk of stream.source) {
      let data: Uint8Array;
      if (chunk instanceof Uint8Array) {
        data = chunk;
      } else if (typeof chunk.subarray === 'function') {
        data = Uint8Array.from(chunk.subarray());
      } else {
        data = Uint8Array.from(chunk as Iterable<number>);
      }
      const newBuffer = new Uint8Array(buffer.length + data.length);
      newBuffer.set(buffer);
      newBuffer.set(data, buffer.length);
      buffer = newBuffer;

      // Process complete messages
      while (true) {
        const result = unframeMessage(buffer);
        if (!result) break;

        buffer = Uint8Array.from(result.remaining);
        const msg = cborg.decode(result.data) as PX1Message;

        const response = await this.handleMessage(msg, remotePeerId);
        if (response) {
          await this.sendToStream(stream, response);
        }
      }
    }
  }

  /**
   * Handle a PX-1 message
   */
  private async handleMessage(msg: PX1Message, remotePeerId: string): Promise<PX1Message | null> {
    switch (msg.type) {
      case PX1MessageType.PX_RESOLVE_REQ:
        return this.handleResolveRequest(msg as PXResolveRequest, remotePeerId);

      case PX1MessageType.PX_RESOLVE_RESP:
        this.handleResolveResponse(msg as PXResolveResponse);
        return null;

      case PX1MessageType.PX_PUSH:
        this.handlePush(msg as PXPush, remotePeerId);
        return null;

      default:
        console.warn('Unknown PX-1 message type:', (msg as PX1Message).type);
        return null;
    }
  }

  /**
   * Handle resolve request - look up a principal's addresses
   */
  private handleResolveRequest(req: PXResolveRequest, requesterPeerId: string): PXResolveResponse {
    const entry = this.peerCache.get(req.principal);

    // Check if we're allowed to share this peer's info
    if (entry && !this.canSharePeerInfo(entry, requesterPeerId)) {
      return {
        type: PX1MessageType.PX_RESOLVE_RESP,
        requestId: req.requestId,
        peer: null,
        error: 'Not authorized to view this peer',
      };
    }

    if (entry && this.isCacheEntryValid(entry)) {
      return {
        type: PX1MessageType.PX_RESOLVE_RESP,
        requestId: req.requestId,
        peer: {
          principal: entry.principal,
          multiaddrs: entry.multiaddrs,
          lastSeen: entry.lastSeen,
          peerId: entry.peerId,
        },
      };
    }

    return {
      type: PX1MessageType.PX_RESOLVE_RESP,
      requestId: req.requestId,
      peer: null,
    };
  }

  /**
   * Handle resolve response
   */
  private handleResolveResponse(resp: PXResolveResponse): void {
    this.emit('resolve:response', resp);

    if (resp.peer) {
      // Add to cache (as unverified)
      this.addToCache({
        ...resp.peer,
        source: 'resolve',
        cachedAt: Date.now(),
        verified: false,
      });
    }
  }

  /**
   * Handle peer push
   */
  private handlePush(push: PXPush, sourcePeerId: string): void {
    for (const peer of push.peers) {
      this.addToCache({
        ...peer,
        source: sourcePeerId,
        cachedAt: Date.now(),
        verified: false,
      });
    }

    this.emit('peers:received', push.peers);
  }

  /**
   * Add a peer to the cache
   */
  private addToCache(entry: PeerCacheEntry): void {
    const existing = this.peerCache.get(entry.principal);

    // Only update if newer or doesn't exist
    if (!existing || entry.lastSeen > existing.lastSeen) {
      this.peerCache.set(entry.principal, entry);
      this.emit('cache:updated', entry);
    }
  }

  /**
   * Check if we can share a peer's info with another peer
   */
  private canSharePeerInfo(entry: PeerCacheEntry, requesterPeerId: string): boolean {
    // For now, use our global visibility mode
    // In the future, each peer could have their own visibility preference stored
    switch (this.config.visibility) {
      case PeerVisibility.PUBLIC:
        return true;
      case PeerVisibility.PRIVATE:
        // Check if requester is in our allowlist
        // For now, we'd need a mapping of peerId -> principal
        return false; // TODO: implement proper allowlist check
      case PeerVisibility.STEALTH:
        return false;
      default:
        return false;
    }
  }

  /**
   * Check if a cache entry is still valid
   */
  private isCacheEntryValid(entry: PeerCacheEntry): boolean {
    return Date.now() - entry.cachedAt < this.config.cacheTTL;
  }

  /**
   * Clean up expired cache entries
   */
  private cleanupCache(): void {
    for (const [principal, entry] of this.peerCache.entries()) {
      if (!this.isCacheEntryValid(entry)) {
        this.peerCache.delete(principal);
      }
    }
  }

  /**
   * Send a message to a stream
   */
  private async sendToStream(stream: Stream, msg: PX1Message): Promise<void> {
    const encoded = cborg.encode(msg);
    const framed = frameMessage(encoded);

    const outgoing = pushable<Uint8Array>();
    outgoing.push(framed);
    outgoing.end();

    await pipe(outgoing, stream.sink);
  }

  // ===== Public API =====

  /**
   * Resolve a principal's addresses
   */
  async resolve(principal: string, throughPeerId: string): Promise<PeerAddressInfo | null> {
    // Check local cache first
    const cached = this.peerCache.get(principal);
    if (cached && this.isCacheEntryValid(cached)) {
      return {
        principal: cached.principal,
        multiaddrs: cached.multiaddrs,
        lastSeen: cached.lastSeen,
        peerId: cached.peerId,
      };
    }

    // Ask the peer
    const { peerIdFromString } = await import('@libp2p/peer-id');
    const remotePeer = peerIdFromString(throughPeerId);

    try {
      const stream = await this.node.dialProtocol(remotePeer, PX1_PROTOCOL);

      const requestId = Buffer.from(randomBytes(16)).toString('hex');
      const request: PXResolveRequest = {
        type: PX1MessageType.PX_RESOLVE_REQ,
        principal,
        requestId,
      };

      // Send request
      await this.sendToStream(stream, request);

      // Wait for response
      return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
          cleanup();
          reject(new Error('Resolve timeout'));
        }, 10000);

        const onResponse = (resp: PXResolveResponse) => {
          if (resp.requestId === requestId) {
            cleanup();
            resolve(resp.peer);
          }
        };

        const cleanup = () => {
          clearTimeout(timeout);
          this.off('resolve:response', onResponse);
          stream.close();
        };

        this.on('resolve:response', onResponse);

        // Read response
        (async () => {
          let buffer = new Uint8Array(0);
          for await (const chunk of stream.source) {
            const data = chunk instanceof Uint8Array ? chunk : new Uint8Array(chunk.subarray());
            const newBuffer = new Uint8Array(buffer.length + data.length);
            newBuffer.set(buffer);
            newBuffer.set(data, buffer.length);
            buffer = newBuffer;

            const result = unframeMessage(buffer);
            if (result) {
              const msg = cborg.decode(result.data) as PX1Message;
              if (msg.type === PX1MessageType.PX_RESOLVE_RESP) {
                onResponse(msg as PXResolveResponse);
                break;
              }
            }
          }
        })().catch(reject);
      });
    } catch (err) {
      console.error('PX-1 resolve failed:', err);
      return null;
    }
  }

  /**
   * Push peer information to a connected peer
   */
  async pushPeers(toPeerId: string, peers?: PeerAddressInfo[]): Promise<void> {
    // If no peers specified, share what we know (respecting visibility)
    if (!peers) {
      peers = this.getShareablePeers(toPeerId);
    }

    if (peers.length === 0) return;

    // Limit to max push peers
    if (peers.length > this.config.maxPushPeers) {
      peers = peers.slice(0, this.config.maxPushPeers);
    }

    const { peerIdFromString } = await import('@libp2p/peer-id');
    const remotePeer = peerIdFromString(toPeerId);

    try {
      const stream = await this.node.dialProtocol(remotePeer, PX1_PROTOCOL);

      const push: PXPush = {
        type: PX1MessageType.PX_PUSH,
        peers,
      };

      await this.sendToStream(stream, push);
      stream.close();
    } catch (err) {
      console.error('PX-1 push failed:', err);
    }
  }

  /**
   * Get peers that can be shared with a given peer
   */
  private getShareablePeers(requesterPeerId: string): PeerAddressInfo[] {
    const shareable: PeerAddressInfo[] = [];

    // Add our own info
    const ourMultiaddrs = this.node.getMultiaddrs().map(ma => ma.toString());
    if (ourMultiaddrs.length > 0) {
      shareable.push({
        principal: this.ourPrincipal,
        multiaddrs: ourMultiaddrs,
        lastSeen: Date.now(),
        peerId: this.ourPeerId,
      });
    }

    // Add cached peers (if visibility allows)
    for (const entry of this.peerCache.values()) {
      if (this.isCacheEntryValid(entry) && this.canSharePeerInfo(entry, requesterPeerId)) {
        shareable.push({
          principal: entry.principal,
          multiaddrs: entry.multiaddrs,
          lastSeen: entry.lastSeen,
          peerId: entry.peerId,
        });
      }
    }

    return shareable;
  }

  /**
   * Update our own peer info after connecting
   */
  updateOwnInfo(principal: string, peerId: string): void {
    this.ourPrincipal = principal;
    this.ourPeerId = peerId;
  }

  /**
   * Add a verified peer (one we've directly connected to)
   */
  addVerifiedPeer(principal: string, peerId: string, multiaddrs: string[]): void {
    this.addToCache({
      principal,
      multiaddrs,
      lastSeen: Date.now(),
      peerId,
      source: 'direct',
      cachedAt: Date.now(),
      verified: true,
    });
  }

  /**
   * Get all known peers
   */
  getKnownPeers(): PeerAddressInfo[] {
    return Array.from(this.peerCache.values())
      .filter(entry => this.isCacheEntryValid(entry))
      .map(entry => ({
        principal: entry.principal,
        multiaddrs: entry.multiaddrs,
        lastSeen: entry.lastSeen,
        peerId: entry.peerId,
      }));
  }

  /**
   * Get a peer's info by principal
   */
  getPeer(principal: string): PeerAddressInfo | null {
    const entry = this.peerCache.get(principal);
    if (!entry || !this.isCacheEntryValid(entry)) {
      return null;
    }
    return {
      principal: entry.principal,
      multiaddrs: entry.multiaddrs,
      lastSeen: entry.lastSeen,
      peerId: entry.peerId,
    };
  }

  /**
   * Set visibility mode
   */
  setVisibility(visibility: PeerVisibility): void {
    this.config.visibility = visibility;
  }

  /**
   * Update allowlist
   */
  setAllowlist(principals: string[]): void {
    this.config.allowlist = principals;
  }
}
