/**
 * libp2p node configuration for SNaP2P
 *
 * Provides:
 * - TCP and WebSocket transports
 * - Noise encryption (libp2p native)
 * - yamux multiplexing
 * - Circuit relay v2 for NAT traversal
 * - DCUtR for hole punching
 * - AutoNAT for NAT detection
 */

import { createLibp2p, type Libp2p } from 'libp2p';
import { tcp } from '@libp2p/tcp';
import { webSockets } from '@libp2p/websockets';
import { noise } from '@chainsafe/libp2p-noise';
import { yamux } from '@chainsafe/libp2p-yamux';
import { circuitRelayTransport, circuitRelayServer } from '@libp2p/circuit-relay-v2';
import { dcutr } from '@libp2p/dcutr';
import { autoNAT } from '@libp2p/autonat';
import { identify } from '@libp2p/identify';
import { bootstrap } from '@libp2p/bootstrap';
import { generateKeyPairFromSeed } from '@libp2p/crypto/keys';
import { peerIdFromPrivateKey } from '@libp2p/peer-id';
import type { FullIdentity } from '../identity/keys.js';

// Protocol identifiers
export const SNAP2P_PROTOCOL = '/snap2p/1.0.0';
export const PX1_PROTOCOL = '/snap2p/px1/1.0.0';

// Default bootstrap nodes (IPFS public bootstrap nodes)
// These help with initial peer discovery and relay
const DEFAULT_BOOTSTRAP_NODES = [
  '/dnsaddr/bootstrap.libp2p.io/p2p/QmNnooDu7bfjPFoTZYxMNLWUQJyrVwtbZg5gBMjTezGAJN',
  '/dnsaddr/bootstrap.libp2p.io/p2p/QmQCU2EcMqAqQPR2i9bChDtGNJchTbq5TbXJJ16u19uLTa',
  '/dnsaddr/bootstrap.libp2p.io/p2p/QmbLHAnMoJPWSCR5Zhtx6BHJX9KiKNN6tpvbUcqanj75Nb',
  '/dnsaddr/bootstrap.libp2p.io/p2p/QmcZf59bWwK5XFi76CZX8cbJ4BhTzzA3gU1ZjYZcYW3dwt',
];

export interface LibP2PNodeConfig {
  identity: FullIdentity;
  listenAddrs?: string[];
  bootstrapNodes?: string[];
  enableRelay?: boolean;
  enableRelayServer?: boolean;
}

export interface LibP2PNode {
  node: Libp2p;
  peerId: string;
  multiaddrs: string[];
}

/**
 * Create a libp2p peer ID from the SNaP2P identity
 *
 * We use the Ed25519 node key from the identity to derive the libp2p peer ID.
 * This ensures consistent identity across the network.
 */
async function createPeerIdFromIdentity(identity: FullIdentity) {
  // libp2p expects a specific key format, we use generateKeyPairFromSeed
  // with our Ed25519 private key as the seed
  const privateKey = await generateKeyPairFromSeed('Ed25519', identity.privateKey);
  const peerId = peerIdFromPrivateKey(privateKey);
  return { privateKey, peerId };
}

/**
 * Create and start a libp2p node
 */
export async function createLibP2PNode(config: LibP2PNodeConfig): Promise<LibP2PNode> {
  const { privateKey, peerId } = await createPeerIdFromIdentity(config.identity);

  const listenAddrs = config.listenAddrs ?? [
    '/ip4/0.0.0.0/tcp/0',
    '/ip4/0.0.0.0/tcp/0/ws',
  ];

  const bootstrapNodes = config.bootstrapNodes ?? DEFAULT_BOOTSTRAP_NODES;

  // Create the node with proper configuration
  const node = await createLibp2p({
    privateKey,
    addresses: {
      listen: listenAddrs,
    },
    transports: [
      tcp(),
      webSockets(),
      // Circuit relay transport for NAT traversal
      ...(config.enableRelay !== false ? [circuitRelayTransport()] : []),
    ],
    connectionEncrypters: [noise()],
    streamMuxers: [yamux()],
    peerDiscovery: bootstrapNodes.length > 0 ? [
      bootstrap({
        list: bootstrapNodes,
      }),
    ] : [],
    services: {
      identify: identify(),
      autoNAT: autoNAT(),
      dcutr: dcutr(),
      // Add relay server if enabled (allows this node to relay for others)
      ...(config.enableRelayServer ? { circuitRelayServer: circuitRelayServer() } : {}),
    },
  });

  await node.start();

  // Get the actual multiaddrs we're listening on
  const multiaddrs = node.getMultiaddrs().map(ma => ma.toString());

  return {
    node,
    peerId: peerId.toString(),
    multiaddrs,
  };
}

/**
 * Stop a libp2p node
 */
export async function stopLibP2PNode(libp2pNode: LibP2PNode): Promise<void> {
  await libp2pNode.node.stop();
}

/**
 * Get current multiaddrs (may change as we discover our public address)
 */
export function getMultiaddrs(libp2pNode: LibP2PNode): string[] {
  return libp2pNode.node.getMultiaddrs().map(ma => ma.toString());
}

/**
 * Dial a peer by multiaddr
 */
export async function dialPeer(libp2pNode: LibP2PNode, multiaddr: string): Promise<void> {
  const { multiaddr: ma } = await import('@multiformats/multiaddr');
  await libp2pNode.node.dial(ma(multiaddr));
}

/**
 * Check if we're connected to a peer
 */
export function isConnected(libp2pNode: LibP2PNode, peerId: string): boolean {
  const connections = libp2pNode.node.getConnections();
  return connections.some(conn => conn.remotePeer.toString() === peerId);
}

/**
 * Get list of connected peers
 */
export function getConnectedPeers(libp2pNode: LibP2PNode): string[] {
  const connections = libp2pNode.node.getConnections();
  return [...new Set(connections.map(conn => conn.remotePeer.toString()))];
}
