/**
 * Stacks wallet implementation per SNaP2P SPECS.md Section 2.2
 *
 * Two-tier key model:
 * - Wallet key: Stacks secp256k1 keypair (signs attestations)
 * - Node key: Ed25519 keypair (used for transport identity)
 *
 * Uses @stacks/wallet-sdk for BIP39 and key derivation.
 */

import {
  generateSecretKey,
  generateWallet as generateStacksWallet,
} from '@stacks/wallet-sdk';
import {
  privateKeyToPublic,
  publicKeyToHex,
  signMessageHashRsv,
  publicKeyToAddress,
  publicKeyFromSignatureRsv,
  getAddressFromPrivateKey,
} from '@stacks/transactions';
import type { StacksNetworkName } from '@stacks/network';
import { sha256 } from '@noble/hashes/sha256';

/**
 * Wallet interface for signing attestations
 * Per SPECS 2.2 - the "Wallet key"
 */
export interface Wallet {
  /** Stacks address (ST... or SP...) */
  readonly address: string;
  /** Full principal (stacks:<address>) */
  readonly principal: string;
  /** Compressed secp256k1 public key (hex) */
  readonly publicKeyHex: string;
  /** Sign a message (returns RSV signature) */
  sign(message: Uint8Array): Promise<Uint8Array>;
}

/**
 * Internal wallet with private key access
 */
export interface WalletWithPrivateKey extends Wallet {
  /** The seed phrase (24 words) */
  readonly mnemonic: string;
  /** Private key (hex with 01 suffix) */
  readonly privateKeyHex: string;
}

/**
 * Generate a new 24-word BIP39 seed phrase
 * Per SPECS - uses 256 bits of entropy
 */
export function generateSeedPhrase(): string {
  return generateSecretKey(256);
}

/**
 * Validate a BIP39 seed phrase
 */
export function isValidSeedPhrase(mnemonic: string): boolean {
  try {
    // Try to generate wallet - will throw if invalid
    const words = mnemonic.trim().split(/\s+/);
    if (words.length !== 24) return false;
    // The wallet-sdk will validate during generation
    return true;
  } catch {
    return false;
  }
}

/**
 * Create wallet from seed phrase
 * Per SPECS 2.2 - derives Stacks address using BIP44 path
 */
export async function walletFromSeedPhrase(
  mnemonic: string,
  testnet = false
): Promise<WalletWithPrivateKey> {
  // Validate word count
  const words = mnemonic.trim().split(/\s+/);
  if (words.length !== 24) {
    throw new Error('Seed phrase must be 24 words');
  }

  // Derive Stacks wallet using wallet-sdk
  // This uses proper BIP44 derivation path for Stacks
  const stacksWallet = await generateStacksWallet({
    secretKey: mnemonic,
    password: mnemonic, // wallet-sdk requires this
  });

  const account = stacksWallet.accounts[0];
  const privateKeyHex = account.stxPrivateKey;

  // Derive address using proper Stacks methods
  const network: StacksNetworkName = testnet ? 'testnet' : 'mainnet';
  const address = getAddressFromPrivateKey(privateKeyHex, network);

  // Get public key
  const publicKeyBytes = privateKeyToPublic(privateKeyHex);
  const publicKeyHex = publicKeyToHex(publicKeyBytes);

  return {
    mnemonic,
    address,
    principal: `stacks:${address}`,
    publicKeyHex,
    privateKeyHex,
    async sign(message: Uint8Array): Promise<Uint8Array> {
      // Hash the message per SPECS
      const messageHash = sha256(message);
      const messageHashHex = Buffer.from(messageHash).toString('hex');

      // Sign with RSV format
      const signatureHex = signMessageHashRsv({
        privateKey: privateKeyHex,
        messageHash: messageHashHex,
      });

      return hexToBytes(signatureHex);
    },
  };
}

/**
 * Generate a new wallet with fresh seed phrase
 */
export async function generateWallet(testnet = false): Promise<WalletWithPrivateKey> {
  const mnemonic = generateSeedPhrase();
  return walletFromSeedPhrase(mnemonic, testnet);
}

/**
 * Verify a wallet signature by recovering the public key
 * Per SPECS 2.3.1 - recovers address from signature and compares
 */
export function verifyWalletSignature(
  message: Uint8Array,
  signature: Uint8Array,
  expectedAddress: string,
  testnet = false
): boolean {
  try {
    const messageHash = sha256(message);
    const messageHashHex = Buffer.from(messageHash).toString('hex');
    const signatureHex = bytesToHex(signature);

    // Recover public key from RSV signature
    const recoveredPublicKeyHex = publicKeyFromSignatureRsv(messageHashHex, signatureHex);

    // Derive address from recovered public key
    const network: StacksNetworkName = testnet ? 'testnet' : 'mainnet';
    const recoveredAddress = publicKeyToAddress(recoveredPublicKeyHex, network);

    return recoveredAddress === expectedAddress;
  } catch {
    return false;
  }
}

/**
 * Get Stacks address from public key
 */
export function addressFromPublicKey(publicKeyHex: string, testnet = false): string {
  const network: StacksNetworkName = testnet ? 'testnet' : 'mainnet';
  return publicKeyToAddress(publicKeyHex, network);
}

// Utility functions
function hexToBytes(hex: string): Uint8Array {
  const bytes = new Uint8Array(hex.length / 2);
  for (let i = 0; i < bytes.length; i++) {
    bytes[i] = parseInt(hex.substr(i * 2, 2), 16);
  }
  return bytes;
}

function bytesToHex(bytes: Uint8Array): string {
  return Array.from(bytes)
    .map(b => b.toString(16).padStart(2, '0'))
    .join('');
}

export { bytesToHex, hexToBytes };
