/**
 * Identity and attestation management per SNaP2P SPECS.md
 *
 * Two-tier key model (SPECS 2.2):
 * - Wallet key: Stacks secp256k1 (signs attestations, proves principal ownership)
 * - Node key: Ed25519 (transport identity, bound to wallet via attestation)
 *
 * Attestation (SPECS 2.3.1):
 * - Signed by wallet key (secp256k1)
 * - Binds principal to node public key
 * - Verified by recovering address from signature
 */

import { ed25519 } from '@noble/curves/ed25519';
import { randomBytes } from '@noble/hashes/utils';
import { scrypt } from '@noble/hashes/scrypt';
import { gcm } from '@noble/ciphers/aes';
import * as cborg from 'cborg';
import * as fs from 'fs';
import * as path from 'path';
import type { Identity, NodeKeyAttestation } from '../types.js';
import {
  generateWallet,
  walletFromSeedPhrase,
  isValidSeedPhrase,
  verifyWalletSignature,
  bytesToHex,
  hexToBytes,
  type WalletWithPrivateKey,
} from './wallet.js';

const IDENTITY_FILE = 'identity.enc';
const ATTESTATION_DOMAIN = 'snap2p-nodekey-attestation-v1';

// Scrypt parameters per SNaP2P (N=2^17 for strong security)
const SCRYPT_N = 131072; // 2^17
const SCRYPT_R = 8;
const SCRYPT_P = 1;
const SCRYPT_KEYLEN = 32;

// Configurable data directory (for multi-wallet support)
let customDataDir: string | null = null;

/**
 * Set a custom data directory (for multi-wallet support)
 * Call this before any other identity functions
 */
export function setDataDir(dir: string): void {
  customDataDir = dir;
}

/**
 * Get the current data directory path
 * Checks environment variable first (for per-identity storage in gateway mode)
 */
export function getDataDirPath(): string {
  // Check environment variable first (for per-identity storage)
  if (process.env.CLAWCHAT_DATA_DIR) {
    return process.env.CLAWCHAT_DATA_DIR;
  }

  if (customDataDir) {
    return customDataDir;
  }

  return path.join(process.env.HOME || '~', '.clawchat');
}

export function getDataDir(): string {
  const dir = getDataDirPath();
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
  return dir;
}

// Generate Ed25519 node keypair (for transport)
export function generateNodeKeyPair(): { publicKey: Uint8Array; privateKey: Uint8Array } {
  const privateKey = randomBytes(32);
  const publicKey = ed25519.getPublicKey(privateKey);
  return { publicKey, privateKey };
}

/**
 * Full identity: Stacks wallet + Ed25519 node key
 * Per SPECS 2.2 - two-tier key model
 */
export interface FullIdentity extends Identity {
  /** BIP39 seed phrase (24 words) - MUST be backed up */
  mnemonic: string;
  /** Stacks secp256k1 public key (hex) */
  walletPublicKeyHex: string;
  /** Stacks secp256k1 private key (hex) */
  walletPrivateKeyHex: string;
  /** Whether this is a testnet identity */
  testnet: boolean;
  /** Optional nickname for display */
  nick?: string;
}

/**
 * Create new identity with generated seed phrase
 */
export async function createIdentity(testnet = false): Promise<FullIdentity> {
  const wallet = await generateWallet(testnet);
  const nodeKeys = generateNodeKeyPair();

  return {
    principal: wallet.principal,
    address: wallet.address,
    publicKey: nodeKeys.publicKey,      // Ed25519 for transport
    privateKey: nodeKeys.privateKey,
    mnemonic: wallet.mnemonic,
    walletPublicKeyHex: wallet.publicKeyHex,
    walletPrivateKeyHex: wallet.privateKeyHex,
    testnet,
  };
}

/**
 * Recover identity from existing seed phrase
 * Per SPECS - same principal, new node key for this device
 */
export async function recoverIdentity(mnemonic: string, testnet = false): Promise<FullIdentity> {
  const wallet = await walletFromSeedPhrase(mnemonic, testnet);
  const nodeKeys = generateNodeKeyPair(); // New node key for this device

  return {
    principal: wallet.principal,
    address: wallet.address,
    publicKey: nodeKeys.publicKey,
    privateKey: nodeKeys.privateKey,
    mnemonic: wallet.mnemonic,
    walletPublicKeyHex: wallet.publicKeyHex,
    walletPrivateKeyHex: wallet.privateKeyHex,
    testnet,
  };
}

// Derive encryption key from password using scrypt
function deriveKey(password: string, salt: Uint8Array): Uint8Array {
  return scrypt(password, salt, {
    N: SCRYPT_N,
    r: SCRYPT_R,
    p: SCRYPT_P,
    dkLen: SCRYPT_KEYLEN,
  });
}

/**
 * Encrypt and save identity (including seed phrase)
 * Per SNaP2P - uses scrypt + AES-256-GCM
 */
export function saveIdentity(identity: FullIdentity, password: string): void {
  if (password.length < 12) {
    throw new Error('Password must be at least 12 characters');
  }

  const dataDir = getDataDir();
  const filePath = path.join(dataDir, IDENTITY_FILE);

  const plaintext = JSON.stringify({
    principal: identity.principal,
    address: identity.address,
    publicKey: bytesToHex(identity.publicKey),
    privateKey: bytesToHex(identity.privateKey),
    mnemonic: identity.mnemonic,
    walletPublicKeyHex: identity.walletPublicKeyHex,
    walletPrivateKeyHex: identity.walletPrivateKeyHex,
    testnet: identity.testnet,
    nick: identity.nick,
  });

  const salt = randomBytes(16);
  const iv = randomBytes(12);
  const key = deriveKey(password, salt);

  const cipher = gcm(key, iv);
  const ciphertext = cipher.encrypt(new TextEncoder().encode(plaintext));

  // Format: version(1) + salt(16) + iv(12) + ciphertext
  const output = new Uint8Array(1 + 16 + 12 + ciphertext.length);
  output[0] = 2; // version 2 (full Stacks wallet)
  output.set(salt, 1);
  output.set(iv, 17);
  output.set(ciphertext, 29);

  fs.writeFileSync(filePath, Buffer.from(output), { mode: 0o600 });
}

/**
 * Decrypt and load identity
 */
export function loadIdentity(password: string): FullIdentity | null {
  const filePath = path.join(getDataDirPath(), IDENTITY_FILE);

  if (!fs.existsSync(filePath)) {
    return null;
  }

  const data = new Uint8Array(fs.readFileSync(filePath));

  const version = data[0];
  if (version !== 2) {
    throw new Error(`Unsupported identity file version: ${version}. Please recreate your identity.`);
  }

  const salt = data.slice(1, 17);
  const iv = data.slice(17, 29);
  const ciphertext = data.slice(29);

  const key = deriveKey(password, salt);

  try {
    const cipher = gcm(key, iv);
    const plaintext = cipher.decrypt(ciphertext);
    const parsed = JSON.parse(new TextDecoder().decode(plaintext));

    return {
      principal: parsed.principal,
      address: parsed.address,
      publicKey: hexToBytes(parsed.publicKey),
      privateKey: hexToBytes(parsed.privateKey),
      mnemonic: parsed.mnemonic,
      walletPublicKeyHex: parsed.walletPublicKeyHex,
      walletPrivateKeyHex: parsed.walletPrivateKeyHex,
      testnet: parsed.testnet ?? true,
      nick: parsed.nick,
    };
  } catch {
    throw new Error('Invalid password or corrupted identity file');
  }
}

export function identityExists(): boolean {
  return fs.existsSync(path.join(getDataDirPath(), IDENTITY_FILE));
}

/**
 * Update the nickname in an existing identity
 */
export function setNick(password: string, nick: string | undefined): void {
  const identity = loadIdentity(password);
  if (!identity) {
    throw new Error('No identity found');
  }
  identity.nick = nick;
  saveIdentity(identity, password);
}

/**
 * Format a principal with optional nick for display
 * Returns "principal(nick)" if nick is set, otherwise just "principal"
 */
export function formatPrincipalWithNick(principal: string, nick?: string): string {
  if (nick) {
    return `${principal}(${nick})`;
  }
  return principal;
}

// Ed25519 signing (for node key operations)
export function sign(privateKey: Uint8Array, message: Uint8Array): Uint8Array {
  return ed25519.sign(message, privateKey);
}

export function verify(publicKey: Uint8Array, message: Uint8Array, signature: Uint8Array): boolean {
  try {
    return ed25519.verify(signature, message, publicKey);
  } catch {
    return false;
  }
}

/**
 * Create NodeKeyAttestation per SPECS 2.3.1
 *
 * The attestation binds the Stacks principal to the Ed25519 node key.
 * It is signed by the wallet key (secp256k1) to prove principal ownership.
 */
export async function createAttestation(
  identity: FullIdentity,
  validitySeconds = 86400
): Promise<NodeKeyAttestation> {
  // Get wallet for signing
  const wallet = await walletFromSeedPhrase(identity.mnemonic, identity.testnet);

  const now = BigInt(Math.floor(Date.now() / 1000));
  const expiresAt = now + BigInt(validitySeconds);
  const nonce = randomBytes(32); // 32 bytes per SPECS

  // Create canonical payload for signing (CBOR encoded per SPECS)
  const payload = {
    v: 1,
    p: identity.principal,
    npk: bytesToHex(identity.publicKey),
    ts: now,
    exp: expiresAt,
    nonce,
    domain: ATTESTATION_DOMAIN,
  };

  const payloadBytes = cborg.encode(payload);

  // Sign with wallet key (secp256k1)
  const signature = await wallet.sign(payloadBytes);

  return {
    version: 1,
    principal: identity.principal,
    nodePublicKey: identity.publicKey,
    issuedAt: Number(now),
    expiresAt: Number(expiresAt),
    nonce,
    domain: ATTESTATION_DOMAIN,
    signature,
  };
}

/**
 * Verify NodeKeyAttestation per SPECS 2.3.1
 *
 * Verifies:
 * - Structure (version, domain, nonce length)
 * - Timestamps (not expired, clock skew tolerance)
 * - Signature (recovers address and compares to principal)
 */
export function verifyAttestation(
  attestation: NodeKeyAttestation,
  testnet = false
): boolean {
  const now = Math.floor(Date.now() / 1000);
  const clockSkewTolerance = 5 * 60; // 5 minutes per SPECS 2.6

  // Check version
  if (attestation.version !== 1) {
    return false;
  }

  // Check domain
  if (attestation.domain !== ATTESTATION_DOMAIN) {
    return false;
  }

  // Check nonce length (16-32 bytes per SPECS)
  if (!attestation.nonce || attestation.nonce.length < 16 || attestation.nonce.length > 32) {
    return false;
  }

  // Check not expired (with clock skew tolerance)
  if (attestation.expiresAt <= now - clockSkewTolerance) {
    return false;
  }

  // Check timestamp not in future (with clock skew tolerance)
  if (attestation.issuedAt > now + clockSkewTolerance) {
    return false;
  }

  // Check node public key length (Ed25519 = 32 bytes)
  if (attestation.nodePublicKey.length !== 32) {
    return false;
  }

  // Recreate the payload that was signed
  const payload = {
    v: attestation.version,
    p: attestation.principal,
    npk: bytesToHex(attestation.nodePublicKey),
    ts: BigInt(attestation.issuedAt),
    exp: BigInt(attestation.expiresAt),
    nonce: attestation.nonce,
    domain: attestation.domain,
  };

  const payloadBytes = cborg.encode(payload);

  // Extract address from principal (stacks:ST... -> ST...)
  const address = attestation.principal.replace('stacks:', '');

  // Verify signature by recovering address
  return verifyWalletSignature(payloadBytes, attestation.signature, address, testnet);
}

/**
 * Serialize attestation to bytes (for wire transmission)
 */
export function serializeAttestation(attestation: NodeKeyAttestation): Uint8Array {
  return cborg.encode({
    v: attestation.version,
    p: attestation.principal,
    npk: bytesToHex(attestation.nodePublicKey),
    ts: BigInt(attestation.issuedAt),
    exp: BigInt(attestation.expiresAt),
    nonce: attestation.nonce,
    domain: attestation.domain,
    sig: attestation.signature,
  });
}

/**
 * Deserialize attestation from bytes
 */
export function deserializeAttestation(data: Uint8Array): NodeKeyAttestation {
  const wire = cborg.decode(data) as {
    v: number;
    p: string;
    npk: string;
    ts: bigint;
    exp: bigint;
    nonce: Uint8Array;
    domain: string;
    sig: Uint8Array;
  };

  if (wire.v !== 1) {
    throw new Error(`Unsupported attestation version: ${wire.v}`);
  }

  return {
    version: 1,
    principal: wire.p,
    nodePublicKey: hexToBytes(wire.npk),
    issuedAt: Number(wire.ts),
    expiresAt: Number(wire.exp),
    nonce: wire.nonce,
    domain: wire.domain,
    signature: wire.sig,
  };
}

// Re-export wallet utilities
export { isValidSeedPhrase as isValidMnemonic } from './wallet.js';
