/**
 * Clawchat Daemon Server
 *
 * Provides:
 * - libp2p-based P2P networking with NAT traversal
 * - SNaP2P authentication layer
 * - PX-1 peer exchange for mesh connectivity
 * - IPC interface for CLI commands
 */

import * as net from 'net';
import * as fs from 'fs';
import * as path from 'path';
import { EventEmitter } from 'events';
import { randomBytes } from '@noble/hashes/utils';
import { getDataDir, type FullIdentity } from '../identity/keys.js';
import type { Message, Peer } from '../types.js';
import {
  createLibP2PNode,
  stopLibP2PNode,
  getMultiaddrs,
  type LibP2PNode,
} from '../net/libp2p-node.js';
import { Snap2pSession, Snap2pProtocolHandler } from '../net/snap2p-protocol.js';
import { PX1Handler } from '../net/px1.js';
import { PeerVisibility, type PeerAddressInfo } from '../types/px1.js';
import { IdentityManager } from './identity-manager.js';
import { isGatewayMode, loadGatewayConfig } from './gateway-config.js';
import type { GatewayConfig, LoadedIdentity } from '../types/gateway.js';
import { MessageRouter } from './message-router.js';

const SOCKET_NAME = 'clawchat.sock';
const PID_FILE = 'daemon.pid';
const INBOX_FILE = 'inbox.json';
const OUTBOX_FILE = 'outbox.json';
const PEERS_FILE = 'peers.json';

export interface DaemonConfig {
  p2pPort: number;
  bootstrapNodes?: string[];
  enableRelay?: boolean;
  enableRelayServer?: boolean;
  gatewayConfig?: GatewayConfig; // Optional: will auto-detect if not provided
}

// IPC command types
export type IpcCommand =
  | { cmd: 'send'; to: string; content: string; as?: string }
  | { cmd: 'recv'; since?: number; timeout?: number; as?: string }
  | { cmd: 'inbox'; as?: string }
  | { cmd: 'outbox'; as?: string }
  | { cmd: 'peers'; as?: string }
  | { cmd: 'peer_add'; principal: string; address: string; alias?: string; as?: string }
  | { cmd: 'peer_remove'; principal: string; as?: string }
  | { cmd: 'peer_resolve'; principal: string; through?: string; as?: string }
  | { cmd: 'status'; as?: string }
  | { cmd: 'multiaddrs' }
  | { cmd: 'connect'; multiaddr: string; as?: string }
  | { cmd: 'stop' };

export interface IpcResponse {
  ok: boolean;
  data?: unknown;
  error?: string;
}

export class Daemon extends EventEmitter {
  private p2pPort: number;
  private config: DaemonConfig;
  private libp2pNode: LibP2PNode | null = null;
  private snap2pHandler: Snap2pProtocolHandler | null = null;
  private px1Handler: PX1Handler | null = null;
  private ipcServer: net.Server | null = null;
  private sessions: Map<string, Snap2pSession> = new Map();
  private dataDir: string;

  // Gateway fields (always present)
  private gatewayConfig: GatewayConfig;
  private identityManager: IdentityManager;
  private messageRouter: MessageRouter;

  constructor(config: DaemonConfig) {
    super();

    // Load or use provided gateway config
    if (config.gatewayConfig) {
      this.gatewayConfig = config.gatewayConfig;
    } else if (isGatewayMode()) {
      this.gatewayConfig = loadGatewayConfig();
    } else {
      throw new Error(
        'No gateway configuration found. Run "clawchat gateway init" to create one.'
      );
    }

    // Initialize gateway components
    this.identityManager = new IdentityManager();
    this.messageRouter = new MessageRouter(this.identityManager);
    this.dataDir = getDataDir();

    this.p2pPort = config.p2pPort;
    this.config = config;
  }


  async start(): Promise<void> {
    // Write PID file
    fs.writeFileSync(
      path.join(this.dataDir, PID_FILE),
      process.pid.toString()
    );

    // Get identity for libp2p node
    const firstIdentity = this.messageRouter.getDefaultIdentity();
    if (!firstIdentity) {
      throw new Error('At least one identity must be loaded to start daemon');
    }
    const nodeIdentity = firstIdentity.identity;

    // Create libp2p node
    this.libp2pNode = await createLibP2PNode({
      identity: nodeIdentity,
      listenAddrs: [
        `/ip4/0.0.0.0/tcp/${this.p2pPort}`,
        `/ip4/0.0.0.0/tcp/${this.p2pPort + 1}/ws`,
      ],
      bootstrapNodes: this.config.bootstrapNodes,
      enableRelay: this.config.enableRelay,
      enableRelayServer: this.config.enableRelayServer,
    });

    // Set up SNaP2P protocol handler with identity resolver
    const identityResolver = (remotePrincipal?: string): FullIdentity | null => {
      if (remotePrincipal) {
        // Find which identities allow this peer
        const allowedIdentities = this.messageRouter.findIdentitiesForPeer(remotePrincipal);
        if (allowedIdentities.length > 0) {
          // Use first allowed identity
          return allowedIdentities[0].identity;
        }
        return null;
      } else {
        // No remote principal yet, use default identity
        const defaultIdentity = this.messageRouter.getDefaultIdentity();
        return defaultIdentity ? defaultIdentity.identity : null;
      }
    };

    this.snap2pHandler = new Snap2pProtocolHandler(
      this.libp2pNode.node,
      identityResolver
    );

    this.snap2pHandler.on('session', (session: Snap2pSession) => {
      this.handleSession(session);
    });

    this.snap2pHandler.on('message', (msg, session) => {
      this.handleMessage(msg, session);
    });

    this.snap2pHandler.on('session:close', (session: Snap2pSession) => {
      if (session.remote) {
        this.sessions.delete(session.remote);
        this.emit('p2p:disconnected', session.remote);
      }
    });

    this.snap2pHandler.start();

    // Set up PX-1 peer exchange
    this.px1Handler = new PX1Handler(
      this.libp2pNode.node,
      nodeIdentity.principal
    );

    this.px1Handler.on('peers:received', (peers: PeerAddressInfo[]) => {
      this.handleReceivedPeers(peers);
    });

    this.px1Handler.start();

    // Start IPC server (unix socket)
    const socketPath = path.join(this.dataDir, SOCKET_NAME);

    // Remove stale socket
    if (fs.existsSync(socketPath)) {
      fs.unlinkSync(socketPath);
    }

    this.ipcServer = net.createServer((socket) => {
      this.handleIpcConnection(socket);
    });

    this.ipcServer.listen(socketPath);

    // Process outbox periodically
    setInterval(() => this.processOutbox(), 5000);

    // Push peers to connected nodes periodically
    setInterval(() => this.pushPeersToConnected(), 60000);

    const multiaddrs = getMultiaddrs(this.libp2pNode);

    const defaultIdentity = this.messageRouter.getDefaultIdentity();
    this.emit('started', {
      p2pPort: this.p2pPort,
      ipcSocket: socketPath,
      principal: defaultIdentity?.identity.principal || 'unknown',
      peerId: this.libp2pNode.peerId,
      multiaddrs,
    });
  }

  private handleSession(session: Snap2pSession): void {
    const remote = session.remote;
    if (!remote) return;

    this.sessions.set(remote, session);

    // Get the identity for this session
    const localPrincipal = session.localPrincipal;
    const identity = this.identityManager.getIdentity(localPrincipal);
    if (!identity) return;

    // Update peer lastSeen and add peerId if we have it
    const peer = identity.peers.find((p: Peer) => p.principal === remote);
    if (peer) {
      peer.lastSeen = Date.now();
      // Store the multiaddrs if available
      if (session.peerId) {
        // The address field now stores multiaddrs
        const existingAddrs = peer.address ? peer.address.split(',') : [];
        const currentAddrs = this.libp2pNode?.node.getConnections()
          .filter(c => c.remotePeer.toString() === session.peerId)
          .map(c => c.remoteAddr.toString()) ?? [];

        if (currentAddrs.length > 0) {
          const allAddrs = [...new Set([...existingAddrs, ...currentAddrs])];
          peer.address = allAddrs.join(',');
        }
      }
      this.identityManager.savePeers(localPrincipal);

      // Add to PX-1 cache as verified
      if (this.px1Handler && session.peerId) {
        const multiaddrs = peer.address?.split(',') ?? [];
        this.px1Handler.addVerifiedPeer(remote, session.peerId, multiaddrs);
      }
    }

    this.emit('p2p:connected', remote);

    // Push our known peers to the new connection
    this.pushPeersToSession(session);
  }

  private handleMessage(msg: { id: string; from: string; nick?: string; content: string; timestamp: number }, session: Snap2pSession): void {
    // Use session's local identity as recipient
    const recipientPrincipal = session.localPrincipal;

    const message: Message = {
      id: msg.id,
      from: msg.from,
      fromNick: msg.nick,
      to: recipientPrincipal,
      content: msg.content,
      timestamp: msg.timestamp,
      status: 'delivered',
    };

    // Route message with ACL enforcement
    const result = this.messageRouter.routeInbound(message, msg.from);
    if (!result.success) {
      console.warn(`[gateway] Message routing failed: ${result.error}`);
      return;
    }

    // Add to target identity's inbox
    this.identityManager.addToInbox(message.to, message);
    this.identityManager.saveInbox(message.to);

    this.emit('message', message);

    // Trigger openclaw system event if enabled for this identity
    if (result.identity!.config.openclawWake) {
      this.triggerOpenclawWake(message);
    }
  }

  private triggerOpenclawWake(message: Message): void {
    try {
      const { spawnSync } = require('child_process');

      // Determine priority based on message content
      const isUrgent = message.content.startsWith('URGENT:') ||
                      message.content.startsWith('ALERT:') ||
                      message.content.startsWith('CRITICAL:');

      const mode = isUrgent ? 'now' : 'next-heartbeat';

      // Format message for openclaw
      const fromDisplay = message.fromNick
        ? `${message.from}(${message.fromNick})`
        : message.from;

      const wakeMessage = `ClawChat from ${fromDisplay}: ${message.content}`;

      // Spawn openclaw system event command
      // Use spawnSync with timeout to avoid blocking
      const result = spawnSync('openclaw', ['system', 'event', '--text', wakeMessage, '--mode', mode], {
        timeout: 5000,  // 5 second timeout
        stdio: 'ignore'  // Don't capture output
      });

      // Log error if command failed (but don't crash daemon)
      if (result.error) {
        console.error('[openclaw-event] Failed to trigger system event:', result.error.message);
      }
    } catch (error) {
      // Silent fail - openclaw might not be installed or available
      console.error('[openclaw-event] Error triggering system event:', error);
    }
  }

  private handleReceivedPeers(peers: PeerAddressInfo[]): void {
    // Add discovered peers to all loaded identities
    const identities = this.identityManager.getAllIdentities();

    for (const identity of identities) {
      for (const peerInfo of peers) {
        // Don't add ourselves
        if (peerInfo.principal === identity.identity.principal) continue;

        const existing = identity.peers.find((p: Peer) => p.principal === peerInfo.principal);
        if (existing) {
          // Merge multiaddrs
          const existingAddrs = existing.address?.split(',') ?? [];
          const newAddrs = [...new Set([...existingAddrs, ...peerInfo.multiaddrs])];
          existing.address = newAddrs.join(',');
          existing.lastSeen = Math.max(existing.lastSeen ?? 0, peerInfo.lastSeen);
        } else {
          // Add new peer
          identity.peers.push({
            principal: peerInfo.principal,
            address: peerInfo.multiaddrs.join(','),
            lastSeen: peerInfo.lastSeen,
          });
        }
      }

      this.identityManager.savePeers(identity.identity.principal);
    }

    this.emit('peers:discovered', peers);
  }

  private async pushPeersToSession(session: Snap2pSession): Promise<void> {
    if (!this.px1Handler || !session.peerId) return;

    try {
      await this.px1Handler.pushPeers(session.peerId);
    } catch (err) {
      console.error('Failed to push peers:', err);
    }
  }

  private async pushPeersToConnected(): Promise<void> {
    if (!this.px1Handler) return;

    for (const session of this.sessions.values()) {
      await this.pushPeersToSession(session);
    }
  }

  private handleIpcConnection(socket: net.Socket): void {
    let buffer = '';

    socket.on('data', async (data) => {
      buffer += data.toString();

      // Simple newline-delimited JSON protocol
      const lines = buffer.split('\n');
      buffer = lines.pop() || '';

      for (const line of lines) {
        if (!line.trim()) continue;

        try {
          const cmd = JSON.parse(line) as IpcCommand;
          const response = await this.handleIpcCommand(cmd);
          socket.write(JSON.stringify(response) + '\n');
        } catch (err) {
          socket.write(JSON.stringify({
            ok: false,
            error: String(err),
          }) + '\n');
        }
      }
    });
  }

  /**
   * Execute an IPC command directly (useful for testing)
   */
  async executeCommand(cmd: IpcCommand): Promise<IpcResponse> {
    return this.handleIpcCommand(cmd);
  }

  /**
   * Get identity from command 'as' field or default
   */
  private getIdentityForCommand(asParam?: string): LoadedIdentity | null {
    if (asParam) {
      // Use specified identity (principal or nickname)
      const identity = this.identityManager.getIdentity(asParam);
      if (!identity) {
        return null;
      }
      return identity;
    } else {
      // Use default identity
      return this.messageRouter.getDefaultIdentity();
    }
  }

  private async handleIpcCommand(cmd: IpcCommand): Promise<IpcResponse> {
    switch (cmd.cmd) {
      case 'send': {
        const id = Buffer.from(randomBytes(16)).toString('hex');

        // Get identity from --as parameter or default
        const sourceIdentity = this.getIdentityForCommand(cmd.as);
        if (!sourceIdentity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }

        const message: Message = {
          id,
          from: sourceIdentity.identity.principal,
          to: cmd.to,
          content: cmd.content,
          timestamp: Date.now(),
          status: 'pending',
        };

        // Add to source identity's outbox
        this.identityManager.addToOutbox(sourceIdentity.identity.principal, message);
        this.identityManager.saveOutbox(sourceIdentity.identity.principal);

        // Try immediate delivery
        await this.tryDeliver(message);

        return { ok: true, data: { id, status: 'queued' } };
      }

      case 'recv': {
        const since = cmd.since || 0;
        const timeout = cmd.timeout;

        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }

        // If no timeout, return immediately
        if (!timeout || timeout <= 0) {
          const messages = identity.inbox.filter(m => m.timestamp > since);
          return { ok: true, data: messages };
        }

        // Wait for messages with timeout
        return await this.recvWithTimeout(since, timeout, identity.identity.principal);
      }

      case 'inbox': {
        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }
        return { ok: true, data: identity.inbox };
      }

      case 'outbox': {
        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }
        return { ok: true, data: identity.outbox };
      }

      case 'peers': {
        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }
        return {
          ok: true,
          data: identity.peers.map(p => ({
            ...p,
            connected: this.sessions.has(p.principal),
          })),
        };
      }

      case 'peer_add': {
        const peer: Peer = {
          principal: cmd.principal,
          address: cmd.address,
          alias: cmd.alias,
          lastSeen: Date.now(),
        };

        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }

        this.identityManager.addOrUpdatePeer(identity.identity.principal, peer);
        this.identityManager.savePeers(identity.identity.principal);

        return { ok: true, data: peer };
      }

      case 'peer_remove': {
        const identity = this.getIdentityForCommand(cmd.as);
        if (!identity) {
          return { ok: false, error: cmd.as ? `Identity not found: ${cmd.as}` : 'No identities loaded' };
        }

        identity.peers = identity.peers.filter(p => p.principal !== cmd.principal);
        this.identityManager.savePeers(identity.identity.principal);

        return { ok: true };
      }

      case 'peer_resolve': {
        if (!this.px1Handler) {
          return { ok: false, error: 'PX-1 not initialized' };
        }

        // If through is specified, ask that peer
        // Otherwise, check our local cache
        const through = cmd.through;
        if (through) {
          const session = this.sessions.get(through);
          if (!session?.peerId) {
            return { ok: false, error: 'Not connected to relay peer' };
          }
          const result = await this.px1Handler.resolve(cmd.principal, session.peerId);
          return { ok: true, data: result };
        } else {
          const cached = this.px1Handler.getPeer(cmd.principal);
          return { ok: true, data: cached };
        }
      }

      case 'status': {
        const identity = this.getIdentityForCommand(cmd.as);
        return {
          ok: true,
          data: {
            principal: identity?.identity.principal || 'unknown',
            peerId: this.libp2pNode?.peerId,
            p2pPort: this.p2pPort,
            multiaddrs: this.libp2pNode ? getMultiaddrs(this.libp2pNode) : [],
            connectedPeers: Array.from(this.sessions.keys()),
            inboxCount: identity?.inbox.length || 0,
            outboxCount: identity?.outbox.filter((m: Message) => m.status === 'pending').length || 0,
            loadedIdentities: this.identityManager.getAllIdentities().map(id => ({
              principal: id.identity.principal,
              nick: id.config.nick,
            })),
          },
        };
      }

      case 'multiaddrs':
        return {
          ok: true,
          data: this.libp2pNode ? getMultiaddrs(this.libp2pNode) : [],
        };

      case 'connect': {
        try {
          if (!this.libp2pNode) {
            return { ok: false, error: 'libp2p not initialized' };
          }

          const { multiaddr } = await import('@multiformats/multiaddr');
          const ma = multiaddr(cmd.multiaddr);
          await this.libp2pNode.node.dial(ma);

          // Extract peer ID from multiaddr if present
          const peerIdStr = ma.getPeerId();
          if (peerIdStr && this.snap2pHandler) {
            // Open SNaP2P stream for authentication
            const session = await this.snap2pHandler.connect(peerIdStr);
            return { ok: true, data: { connected: true, principal: session.remote } };
          }

          return { ok: true, data: { connected: true } };
        } catch (err) {
          return { ok: false, error: String(err) };
        }
      }

      case 'stop':
        await this.stop();
        return { ok: true, data: { status: 'stopping' } };

      default:
        return { ok: false, error: 'Unknown command' };
    }
  }

  /**
   * Wait for messages with a timeout
   * Returns all messages received since `since` timestamp, waiting up to `timeout` ms
   */
  private async recvWithTimeout(since: number, timeout: number, principal: string): Promise<IpcResponse> {
    const startTime = Date.now();
    const endTime = startTime + timeout;

    // Collect messages that arrive during the timeout period
    const collectedMessages: Message[] = [];

    // Get any existing messages first for the specified identity
    const identity = this.identityManager.getIdentity(principal);
    if (identity) {
      const existingMessages = identity.inbox.filter((m: Message) => m.timestamp > since);
      collectedMessages.push(...existingMessages);
    }

    // If we already have messages, we could return immediately
    // But the user wants to wait for the full timeout to catch ACKs, etc.
    // So we'll wait and collect any new messages that arrive

    return new Promise((resolve) => {
      const messageHandler = (msg: Message) => {
        // Only collect messages for the specified identity
        if (msg.timestamp > since && msg.to === principal) {
          collectedMessages.push(msg);
        }
      };

      this.on('message', messageHandler);

      const checkAndResolve = () => {
        this.off('message', messageHandler);
        // Return unique messages (dedupe by id)
        const uniqueMessages = Array.from(
          new Map(collectedMessages.map(m => [m.id, m])).values()
        );
        resolve({ ok: true, data: uniqueMessages });
      };

      // Set timeout to resolve
      setTimeout(checkAndResolve, timeout);
    });
  }

  private async tryDeliver(message: Message): Promise<boolean> {
    // ALIAS RESOLUTION: Check if message.to is an alias and resolve to principal
    // Check all identities' peer lists for alias matches
    if (!message.to.startsWith('stacks:')) {
      const identities = this.identityManager.getAllIdentities();
      for (const identity of identities) {
        const peer = identity.peers.find((p: Peer) => p.alias === message.to);
        if (peer) {
          message.to = peer.principal;
          break;
        }
      }
    }
    
    // LOCAL DELIVERY: Check if recipient is a co-hosted identity in this gateway
    const recipientIdentity = this.identityManager.getIdentity(message.to);
    if (recipientIdentity) {
      // Deliver directly to local identity's inbox
      this.identityManager.addToInbox(message.to, {
        ...message,
        status: 'delivered',
      });
      this.identityManager.saveInbox(message.to);

      // Update sender's outbox status
      message.status = 'sent';
      const senderIdentity = this.identityManager.getIdentity(message.from);
      if (senderIdentity) {
        this.identityManager.saveOutbox(message.from);
      }

      // Emit message event for local delivery
      this.emit('message', { ...message, status: 'delivered' });

      // Trigger openclaw system event if enabled for recipient
      if (recipientIdentity.config.openclawWake) {
        this.triggerOpenclawWake(message);
      }

      return true;
    }

    // Check if we have an active session
    let session = this.sessions.get(message.to);

    if (session && session.isAuthenticated) {
      try {
        await session.sendChatMessage(message.content);
        message.status = 'sent';
        // Save to sender's outbox
        const senderIdentity = this.identityManager.getIdentity(message.from);
        if (senderIdentity) {
          this.identityManager.saveOutbox(message.from);
        }
        return true;
      } catch {
        // Session might be stale, remove it
        this.sessions.delete(message.to);
      }
    }

    // Try to connect to the peer - check all identities for peer info
    const identities = this.identityManager.getAllIdentities();
    let peer: Peer | undefined;
    let resolvedPrincipal = message.to;
    
    for (const identity of identities) {
      // First try to match by principal directly
      peer = identity.peers.find((p: Peer) => p.principal === message.to);
      if (peer) break;
      
      // Then try to match by alias
      peer = identity.peers.find((p: Peer) => p.alias === message.to);
      if (peer) {
        // Update message.to to the resolved principal for future lookups
        resolvedPrincipal = peer.principal;
        message.to = peer.principal;
        break;
      }
    }
    if (!peer?.address) {
      // Try PX-1 resolution through connected peers
      if (this.px1Handler && this.sessions.size > 0) {
        for (const connectedSession of this.sessions.values()) {
          if (connectedSession.peerId) {
            const resolved = await this.px1Handler.resolve(message.to, connectedSession.peerId);
            if (resolved && resolved.multiaddrs.length > 0) {
              // Try to connect via resolved addresses
              for (const addr of resolved.multiaddrs) {
                const connected = await this.tryConnect(addr);
                if (connected) {
                  session = this.sessions.get(message.to);
                  if (session?.isAuthenticated) {
                    try {
                      await session.sendChatMessage(message.content);
                      message.status = 'sent';
                      // Save to sender's outbox
                      const senderIdentity = this.identityManager.getIdentity(message.from);
                      if (senderIdentity) {
                        this.identityManager.saveOutbox(message.from);
                      }
                      return true;
                    } catch {
                      // Continue trying
                    }
                  }
                }
              }
            }
          }
        }
      }
      return false;
    }

    // Try each address (comma-separated multiaddrs)
    const addresses = peer.address.split(',');
    for (const addr of addresses) {
      const connected = await this.tryConnect(addr.trim());
      if (connected) {
        session = this.sessions.get(message.to);
        if (session?.isAuthenticated) {
          try {
            await session.sendChatMessage(message.content);
            message.status = 'sent';
            // Save to sender's outbox
            const senderIdentity = this.identityManager.getIdentity(message.from);
            if (senderIdentity) {
              this.identityManager.saveOutbox(message.from);
            }
            return true;
          } catch {
            // Continue trying
          }
        }
      }
    }

    return false;
  }

  private async tryConnect(address: string): Promise<boolean> {
    if (!this.libp2pNode || !this.snap2pHandler) return false;

    try {
      // Check if it's a multiaddr or legacy host:port
      if (address.startsWith('/')) {
        // It's a multiaddr
        const { multiaddr } = await import('@multiformats/multiaddr');
        const ma = multiaddr(address);
        await this.libp2pNode.node.dial(ma);

        const peerIdStr = ma.getPeerId();
        if (peerIdStr) {
          await this.snap2pHandler.connect(peerIdStr);
          return true;
        }
      } else if (address.includes(':')) {
        // Legacy host:port format - convert to multiaddr
        const [host, port] = address.split(':');
        const ma = `/ip4/${host}/tcp/${port}`;
        const { multiaddr } = await import('@multiformats/multiaddr');
        await this.libp2pNode.node.dial(multiaddr(ma));
        return true;
      }
    } catch (err) {
      console.error(`Failed to connect to ${address}:`, err);
    }

    return false;
  }

  private async processOutbox(): Promise<void> {
    // Process outbox for all loaded identities
    const identities = this.identityManager.getAllIdentities();

    for (const identity of identities) {
      const pending = identity.outbox.filter((m: Message) => m.status === 'pending');

      for (const message of pending) {
        await this.tryDeliver(message);
      }
    }
  }

  async stop(exitProcess = true): Promise<void> {
    // Stop PX-1 handler
    if (this.px1Handler) {
      this.px1Handler.stop();
    }

    // Stop SNaP2P handler
    if (this.snap2pHandler) {
      this.snap2pHandler.stop();
    }

    // Stop libp2p node
    if (this.libp2pNode) {
      await stopLibP2PNode(this.libp2pNode);
    }

    // Close IPC server
    if (this.ipcServer) {
      this.ipcServer.close();
    }

    // Close all sessions
    for (const session of this.sessions.values()) {
      session.close();
    }

    // Remove socket and PID file
    const socketPath = path.join(this.dataDir, SOCKET_NAME);
    if (fs.existsSync(socketPath)) {
      fs.unlinkSync(socketPath);
    }

    const pidPath = path.join(this.dataDir, PID_FILE);
    if (fs.existsSync(pidPath)) {
      fs.unlinkSync(pidPath);
    }

    this.emit('stopped');
    if (exitProcess) {
      process.exit(0);
    }
  }
}

// Check if daemon is running
export function isDaemonRunning(): boolean {
  const dataDir = getDataDir();
  const pidPath = path.join(dataDir, PID_FILE);

  if (!fs.existsSync(pidPath)) {
    return false;
  }

  const pid = parseInt(fs.readFileSync(pidPath, 'utf-8'), 10);

  try {
    process.kill(pid, 0); // Check if process exists
    return true;
  } catch {
    // Process doesn't exist, clean up stale PID file
    fs.unlinkSync(pidPath);
    return false;
  }
}

// IPC client for CLI
export class IpcClient {
  private socketPath: string;

  constructor() {
    this.socketPath = path.join(getDataDir(), SOCKET_NAME);
  }

  async send(cmd: IpcCommand, socketTimeoutMs?: number): Promise<IpcResponse> {
    // If the command has a timeout (like recv with --timeout), extend socket timeout
    const cmdTimeout = 'timeout' in cmd && typeof cmd.timeout === 'number' ? cmd.timeout : 0;
    const timeout = socketTimeoutMs ?? Math.max(5000, cmdTimeout + 2000); // Add 2s buffer

    return new Promise((resolve, reject) => {
      if (!fs.existsSync(this.socketPath)) {
        reject(new Error('Daemon not running. Start with: clawchat daemon start'));
        return;
      }

      const socket = net.createConnection(this.socketPath);
      let response = '';

      socket.on('connect', () => {
        socket.write(JSON.stringify(cmd) + '\n');
      });

      socket.on('data', (data) => {
        response += data.toString();
        if (response.includes('\n')) {
          socket.end();
          resolve(JSON.parse(response.trim()));
        }
      });

      socket.on('error', reject);

      socket.on('timeout', () => {
        socket.destroy();
        reject(new Error('Connection timeout'));
      });

      socket.setTimeout(timeout);
    });
  }
}
