/**
 * Message Router for Gateway Mode (Phase 2)
 *
 * Routes messages between identities with ACL enforcement.
 * Handles both inbound (P2P) and outbound (IPC) message routing.
 */

import type { Message } from '../types.js';
import type { IdentityManager } from './identity-manager.js';
import type { LoadedIdentity } from '../types/gateway.js';

/**
 * Result of routing attempt
 */
export interface RoutingResult {
  /** Whether routing was successful */
  success: boolean;
  /** Error message if routing failed */
  error?: string;
  /** Target identity if routing succeeded */
  identity?: LoadedIdentity;
}

/**
 * MessageRouter handles routing messages to/from identities with ACL enforcement
 */
export class MessageRouter {
  constructor(private identityManager: IdentityManager) {}

  /**
   * Route an inbound message from P2P network to the correct identity
   * Enforces ACL rules (allowedRemotePeers)
   *
   * @param message - The message to route
   * @param fromPrincipal - Principal of the sender (already authenticated via SNaP2P)
   * @returns Routing result with success status and target identity
   */
  routeInbound(message: Message, fromPrincipal: string): RoutingResult {
    // Find target identity by message recipient
    const targetIdentity = this.identityManager.getIdentity(message.to);

    if (!targetIdentity) {
      return {
        success: false,
        error: `No loaded identity found for recipient: ${message.to}`,
      };
    }

    // Enforce ACL: check if sender is allowed
    if (!this.isRemotePeerAllowed(targetIdentity, fromPrincipal)) {
      return {
        success: false,
        error: `Sender ${fromPrincipal} is not in allowedRemotePeers for identity ${message.to}`,
      };
    }

    // Routing successful
    return {
      success: true,
      identity: targetIdentity,
    };
  }

  /**
   * Route an outbound message from IPC to P2P network
   * Finds the correct identity to send from
   *
   * @param message - The message to route
   * @param requestedIdentity - Optional identity to send from (principal or nickname)
   * @returns Routing result with success status and source identity
   */
  routeOutbound(message: Message, requestedIdentity?: string): RoutingResult {
    let sourceIdentity: LoadedIdentity | null = null;

    // If specific identity requested, use it
    if (requestedIdentity) {
      sourceIdentity = this.identityManager.getIdentity(requestedIdentity);
      if (!sourceIdentity) {
        return {
          success: false,
          error: `Requested identity not found: ${requestedIdentity}`,
        };
      }
    }
    // If message.from is set, use that identity
    else if (message.from) {
      sourceIdentity = this.identityManager.getIdentity(message.from);
      if (!sourceIdentity) {
        return {
          success: false,
          error: `Message.from identity not found: ${message.from}`,
        };
      }
    }
    // Otherwise, use first loaded identity (default behavior)
    else {
      const identities = this.identityManager.getAllIdentities();
      if (identities.length === 0) {
        return {
          success: false,
          error: 'No identities loaded',
        };
      }
      sourceIdentity = identities[0];
    }

    // Verify message.from matches the source identity
    if (message.from && message.from !== sourceIdentity.identity.principal) {
      return {
        success: false,
        error: `Message.from (${message.from}) does not match source identity (${sourceIdentity.identity.principal})`,
      };
    }

    return {
      success: true,
      identity: sourceIdentity,
    };
  }

  /**
   * Check if a local IPC connection is allowed to access an identity
   *
   * @param principalOrNick - Identity to access (principal or nickname)
   * @returns true if access is allowed
   */
  isLocalAccessAllowed(principalOrNick: string): boolean {
    const identity = this.identityManager.getIdentity(principalOrNick);
    if (!identity) {
      return false;
    }
    return identity.config.allowLocal;
  }

  /**
   * Check if a remote peer is allowed to send messages to an identity
   *
   * @param identity - Target identity
   * @param peerPrincipal - Principal of the remote peer
   * @returns true if peer is allowed
   */
  private isRemotePeerAllowed(identity: LoadedIdentity, peerPrincipal: string): boolean {
    const { allowedRemotePeers } = identity.config;

    // Check for wildcard (allow all)
    if (allowedRemotePeers.includes('*')) {
      return true;
    }

    // Check if peer is explicitly allowed
    return allowedRemotePeers.includes(peerPrincipal);
  }

  /**
   * Get the default identity (first loaded identity)
   * Used for operations that don't specify an identity
   *
   * @returns The default identity, or null if no identities loaded
   */
  getDefaultIdentity(): LoadedIdentity | null {
    const identities = this.identityManager.getAllIdentities();
    return identities.length > 0 ? identities[0] : null;
  }

  /**
   * Find which identity should handle an incoming connection from a peer
   * This is used during connection establishment to determine routing
   *
   * @param peerPrincipal - Principal of the connecting peer
   * @returns Array of identities that allow this peer (may be empty)
   */
  findIdentitiesForPeer(peerPrincipal: string): LoadedIdentity[] {
    const identities = this.identityManager.getAllIdentities();
    return identities.filter((identity) =>
      this.isRemotePeerAllowed(identity, peerPrincipal)
    );
  }
}
