/**
 * Identity Manager for Gateway Mode
 *
 * Manages multiple identities with per-identity storage isolation.
 * Each identity has its own encrypted identity file, inbox, outbox, and peer list.
 */

import fs from 'fs';
import path from 'path';
import type { FullIdentity } from '../identity/keys.js';
import { getDataDir, loadIdentity as loadIdentityFromFile } from '../identity/keys.js';
import type { Message, Peer } from '../types.js';
import type { LoadedIdentity, IdentityConfig } from '../types/gateway.js';
import { GatewayConfigError } from '../types/gateway.js';

const IDENTITY_FILE = 'identity.enc';
const INBOX_FILE = 'inbox.json';
const OUTBOX_FILE = 'outbox.json';
const PEERS_FILE = 'peers.json';

/**
 * IdentityManager handles loading, unloading, and managing multiple identities
 * in gateway mode. Each identity is stored in its own directory with isolated state.
 */
export class IdentityManager {
  /** Map of principal -> LoadedIdentity */
  private identities: Map<string, LoadedIdentity> = new Map();

  /** Map of nickname -> principal for quick lookup */
  private nicknameMap: Map<string, string> = new Map();

  /**
   * Load an identity from its per-identity directory
   * @param principal - The principal to load (e.g., "stacks:ST1ABC...")
   * @param password - Password to decrypt the identity
   * @param config - Identity configuration
   * @throws GatewayConfigError if identity cannot be loaded
   */
  async loadIdentity(
    principal: string,
    password: string,
    config: IdentityConfig
  ): Promise<void> {
    // Check if already loaded
    if (this.identities.has(principal)) {
      throw new GatewayConfigError(`Identity ${principal} is already loaded`);
    }

    // Check for nickname conflicts
    if (config.nick && this.nicknameMap.has(config.nick)) {
      throw new GatewayConfigError(`Nickname ${config.nick} is already in use`);
    }

    // Get per-identity directory
    const dataDir = path.join(getDataDir(), 'identities', principal);

    // Verify identity directory exists
    if (!fs.existsSync(dataDir)) {
      throw new GatewayConfigError(
        `Identity directory not found: ${dataDir}. Create the identity first.`
      );
    }

    // Verify identity.enc exists
    const identityPath = path.join(dataDir, IDENTITY_FILE);
    if (!fs.existsSync(identityPath)) {
      throw new GatewayConfigError(
        `Identity file not found: ${identityPath}. Create the identity first.`
      );
    }

    // Load and decrypt identity
    // Use environment variable to override data directory temporarily
    const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
    try {
      process.env.CLAWCHAT_DATA_DIR = dataDir;
      const identity = loadIdentityFromFile(password);

      if (!identity) {
        throw new GatewayConfigError(`Failed to load identity from ${identityPath}`);
      }

      // Verify principal matches
      if (identity.principal !== principal) {
        throw new GatewayConfigError(
          `Principal mismatch: expected ${principal}, got ${identity.principal}`
        );
      }

      // Load per-identity state
      const inbox = this.loadInbox(dataDir);
      const outbox = this.loadOutbox(dataDir);
      const peers = this.loadPeers(dataDir);

      // Create LoadedIdentity
      const loadedIdentity: LoadedIdentity = {
        identity,
        config,
        inbox,
        outbox,
        peers,
        dataDir,
      };

      // Register identity
      this.identities.set(principal, loadedIdentity);

      // Register nickname mapping
      if (config.nick) {
        this.nicknameMap.set(config.nick, principal);
      }

      console.log(
        `[gateway] Loaded identity ${principal}${config.nick ? ` (${config.nick})` : ''}`
      );
    } finally {
      // Restore original environment
      if (originalDataDir !== undefined) {
        process.env.CLAWCHAT_DATA_DIR = originalDataDir;
      } else {
        delete process.env.CLAWCHAT_DATA_DIR;
      }
    }
  }

  /**
   * Unload an identity and save its state
   * @param principal - The principal to unload
   */
  unloadIdentity(principal: string): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }

    // Save state before unloading
    this.saveInbox(principal);
    this.saveOutbox(principal);
    this.savePeers(principal);

    // Remove nickname mapping
    if (loaded.config.nick) {
      this.nicknameMap.delete(loaded.config.nick);
    }

    // Remove from loaded identities
    this.identities.delete(principal);

    console.log(
      `[gateway] Unloaded identity ${principal}${loaded.config.nick ? ` (${loaded.config.nick})` : ''}`
    );
  }

  /**
   * Get a loaded identity by principal or nickname
   * @param principalOrNick - Principal or nickname to lookup
   * @returns LoadedIdentity or null if not found
   */
  getIdentity(principalOrNick: string): LoadedIdentity | null {
    // Try direct principal lookup first
    if (this.identities.has(principalOrNick)) {
      return this.identities.get(principalOrNick)!;
    }

    // Try nickname lookup
    const principal = this.nicknameMap.get(principalOrNick);
    if (principal) {
      return this.identities.get(principal) || null;
    }

    return null;
  }

  /**
   * Get all loaded identities
   * @returns Array of LoadedIdentity objects
   */
  getAllIdentities(): LoadedIdentity[] {
    return Array.from(this.identities.values());
  }

  /**
   * Check if an identity is loaded
   * @param principalOrNick - Principal or nickname to check
   */
  isLoaded(principalOrNick: string): boolean {
    return this.getIdentity(principalOrNick) !== null;
  }

  /**
   * Add a message to an identity's inbox
   */
  addToInbox(principal: string, message: Message): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }
    loaded.inbox.push(message);
  }

  /**
   * Add a message to an identity's outbox
   */
  addToOutbox(principal: string, message: Message): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }
    loaded.outbox.push(message);
  }

  /**
   * Add or update a peer in an identity's peer list
   */
  addOrUpdatePeer(principal: string, peer: Peer): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }

    // Find existing peer
    const index = loaded.peers.findIndex((p) => p.principal === peer.principal);
    if (index >= 0) {
      // Update existing peer
      loaded.peers[index] = { ...loaded.peers[index], ...peer };
    } else {
      // Add new peer
      loaded.peers.push(peer);
    }
  }

  /**
   * Save an identity's inbox to disk
   */
  saveInbox(principal: string): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }

    const inboxPath = path.join(loaded.dataDir, INBOX_FILE);
    fs.writeFileSync(inboxPath, JSON.stringify(loaded.inbox, null, 2), { mode: 0o600 });
  }

  /**
   * Save an identity's outbox to disk
   */
  saveOutbox(principal: string): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }

    const outboxPath = path.join(loaded.dataDir, OUTBOX_FILE);
    fs.writeFileSync(outboxPath, JSON.stringify(loaded.outbox, null, 2), { mode: 0o600 });
  }

  /**
   * Save an identity's peer list to disk
   */
  savePeers(principal: string): void {
    const loaded = this.identities.get(principal);
    if (!loaded) {
      throw new GatewayConfigError(`Identity ${principal} is not loaded`);
    }

    const peersPath = path.join(loaded.dataDir, PEERS_FILE);
    fs.writeFileSync(peersPath, JSON.stringify(loaded.peers, null, 2), { mode: 0o600 });
  }

  /**
   * Load inbox from per-identity directory
   */
  private loadInbox(dataDir: string): Message[] {
    const inboxPath = path.join(dataDir, INBOX_FILE);
    if (fs.existsSync(inboxPath)) {
      try {
        return JSON.parse(fs.readFileSync(inboxPath, 'utf-8'));
      } catch (error) {
        console.warn(`[gateway] Failed to load inbox from ${inboxPath}:`, error);
      }
    }
    return [];
  }

  /**
   * Load outbox from per-identity directory
   */
  private loadOutbox(dataDir: string): Message[] {
    const outboxPath = path.join(dataDir, OUTBOX_FILE);
    if (fs.existsSync(outboxPath)) {
      try {
        return JSON.parse(fs.readFileSync(outboxPath, 'utf-8'));
      } catch (error) {
        console.warn(`[gateway] Failed to load outbox from ${outboxPath}:`, error);
      }
    }
    return [];
  }

  /**
   * Load peers from per-identity directory
   */
  private loadPeers(dataDir: string): Peer[] {
    const peersPath = path.join(dataDir, PEERS_FILE);
    if (fs.existsSync(peersPath)) {
      try {
        return JSON.parse(fs.readFileSync(peersPath, 'utf-8'));
      } catch (error) {
        console.warn(`[gateway] Failed to load peers from ${peersPath}:`, error);
      }
    }
    return [];
  }
}
