/**
 * Gateway Configuration Management
 *
 * Handles loading, saving, and validating gateway-config.json
 */

import fs from 'fs';
import path from 'path';
import { getDataDir } from '../identity/keys.js';
import type { GatewayConfig, IdentityConfig } from '../types/gateway.js';
import { GatewayConfigError } from '../types/gateway.js';

const GATEWAY_CONFIG_FILE = 'gateway-config.json';

/**
 * Get the path to the gateway configuration file
 */
function getGatewayConfigPath(): string {
  return path.join(getDataDir(), GATEWAY_CONFIG_FILE);
}

/**
 * Check if gateway mode is enabled
 * Gateway mode is enabled if gateway-config.json exists
 */
export function isGatewayMode(): boolean {
  const configPath = getGatewayConfigPath();
  return fs.existsSync(configPath);
}

/**
 * Load gateway configuration from disk
 * @throws GatewayConfigError if config doesn't exist or is invalid
 */
export function loadGatewayConfig(): GatewayConfig {
  const configPath = getGatewayConfigPath();

  if (!fs.existsSync(configPath)) {
    throw new GatewayConfigError(
      `Gateway config not found at ${configPath}. Run 'clawchat gateway init' to create one.`
    );
  }

  let configData: unknown;
  try {
    const content = fs.readFileSync(configPath, 'utf-8');
    configData = JSON.parse(content);
  } catch (error) {
    throw new GatewayConfigError(
      `Failed to parse gateway config: ${error instanceof Error ? error.message : String(error)}`
    );
  }

  const config = configData as GatewayConfig;
  validateGatewayConfig(config);

  return config;
}

/**
 * Save gateway configuration to disk
 * @throws GatewayConfigError if config is invalid
 */
export function saveGatewayConfig(config: GatewayConfig): void {
  validateGatewayConfig(config);

  const configPath = getGatewayConfigPath();
  const configDir = path.dirname(configPath);

  // Ensure data directory exists
  if (!fs.existsSync(configDir)) {
    fs.mkdirSync(configDir, { recursive: true, mode: 0o700 });
  }

  try {
    const content = JSON.stringify(config, null, 2);
    fs.writeFileSync(configPath, content, { mode: 0o600 });
  } catch (error) {
    throw new GatewayConfigError(
      `Failed to save gateway config: ${error instanceof Error ? error.message : String(error)}`
    );
  }
}

/**
 * Validate gateway configuration
 * @throws GatewayConfigError if config is invalid
 */
export function validateGatewayConfig(config: unknown): asserts config is GatewayConfig {
  if (!config || typeof config !== 'object') {
    throw new GatewayConfigError('Config must be an object');
  }

  const cfg = config as Partial<GatewayConfig>;

  // Validate version
  if (cfg.version !== 1) {
    throw new GatewayConfigError(`Invalid config version: ${cfg.version}. Expected version 1.`);
  }

  // Validate p2pPort
  if (typeof cfg.p2pPort !== 'number') {
    throw new GatewayConfigError('p2pPort must be a number');
  }
  if (cfg.p2pPort < 1 || cfg.p2pPort > 65535) {
    throw new GatewayConfigError(`Invalid p2pPort: ${cfg.p2pPort}. Must be between 1 and 65535.`);
  }

  // Validate identities array
  if (!Array.isArray(cfg.identities)) {
    throw new GatewayConfigError('identities must be an array');
  }

  if (cfg.identities.length === 0) {
    throw new GatewayConfigError('At least one identity must be configured');
  }

  // Track principals and nicknames to detect duplicates
  const seenPrincipals = new Set<string>();
  const seenNicks = new Set<string>();

  for (let i = 0; i < cfg.identities.length; i++) {
    const identity = cfg.identities[i];
    validateIdentityConfig(identity, i);

    // Check for duplicate principals
    if (seenPrincipals.has(identity.principal)) {
      throw new GatewayConfigError(
        `Duplicate principal at index ${i}: ${identity.principal}`
      );
    }
    seenPrincipals.add(identity.principal);

    // Check for duplicate nicknames
    if (identity.nick) {
      if (seenNicks.has(identity.nick)) {
        throw new GatewayConfigError(`Duplicate nickname at index ${i}: ${identity.nick}`);
      }
      seenNicks.add(identity.nick);
    }
  }
}

/**
 * Validate a single identity configuration
 * @throws GatewayConfigError if identity config is invalid
 */
function validateIdentityConfig(identity: unknown, index: number): asserts identity is IdentityConfig {
  if (!identity || typeof identity !== 'object') {
    throw new GatewayConfigError(`Identity at index ${index} must be an object`);
  }

  const id = identity as Partial<IdentityConfig>;

  // Validate principal
  if (typeof id.principal !== 'string' || !id.principal) {
    throw new GatewayConfigError(`Identity at index ${index}: principal must be a non-empty string`);
  }
  if (!id.principal.startsWith('stacks:')) {
    throw new GatewayConfigError(
      `Identity at index ${index}: principal must start with "stacks:" (got: ${id.principal})`
    );
  }

  // Validate optional nick
  if (id.nick !== undefined) {
    if (typeof id.nick !== 'string' || !id.nick) {
      throw new GatewayConfigError(`Identity at index ${index}: nick must be a non-empty string`);
    }
    // Nickname validation: alphanumeric, underscore, hyphen only
    if (!/^[a-zA-Z0-9_-]+$/.test(id.nick)) {
      throw new GatewayConfigError(
        `Identity at index ${index}: nick must contain only letters, numbers, underscores, and hyphens`
      );
    }
  }

  // Validate boolean fields
  if (typeof id.autoload !== 'boolean') {
    throw new GatewayConfigError(`Identity at index ${index}: autoload must be a boolean`);
  }
  if (typeof id.allowLocal !== 'boolean') {
    throw new GatewayConfigError(`Identity at index ${index}: allowLocal must be a boolean`);
  }
  if (typeof id.openclawWake !== 'boolean') {
    throw new GatewayConfigError(`Identity at index ${index}: openclawWake must be a boolean`);
  }

  // Validate allowedRemotePeers
  if (!Array.isArray(id.allowedRemotePeers)) {
    throw new GatewayConfigError(
      `Identity at index ${index}: allowedRemotePeers must be an array`
    );
  }
  for (let j = 0; j < id.allowedRemotePeers.length; j++) {
    const peer = id.allowedRemotePeers[j];
    if (typeof peer !== 'string' || !peer) {
      throw new GatewayConfigError(
        `Identity at index ${index}: allowedRemotePeers[${j}] must be a non-empty string`
      );
    }
    // Allow "*" or principals starting with "stacks:"
    if (peer !== '*' && !peer.startsWith('stacks:')) {
      throw new GatewayConfigError(
        `Identity at index ${index}: allowedRemotePeers[${j}] must be "*" or a principal starting with "stacks:"`
      );
    }
  }
}

/**
 * Create an initial gateway configuration
 * Helper for testing and gateway initialization
 */
export function createInitialGatewayConfig(
  principal: string,
  nick: string | undefined,
  p2pPort: number
): GatewayConfig {
  const config: GatewayConfig = {
    version: 1,
    p2pPort,
    identities: [
      {
        principal,
        nick,
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      },
    ],
  };

  validateGatewayConfig(config);
  return config;
}
