// Simplified Noise-like handshake using ECDH + ChaCha20-Poly1305
// Based on Noise XX pattern but simplified for our use case

import { x25519 } from '@noble/curves/ed25519';
import { chacha20poly1305 } from '@noble/ciphers/chacha';
import { sha256 } from '@noble/hashes/sha256';
import { hkdf } from '@noble/hashes/hkdf';
import { randomBytes } from '@noble/hashes/utils';

const PROTOCOL_NAME = 'ClawChat_XX_25519_ChaChaPoly_SHA256';

export interface KeyPair {
  publicKey: Uint8Array;
  privateKey: Uint8Array;
}

export interface CipherState {
  key: Uint8Array;
  nonce: bigint;
}

export interface HandshakeState {
  localEphemeral: KeyPair;
  localStatic: KeyPair;
  remoteEphemeralPub?: Uint8Array;
  remoteStaticPub?: Uint8Array;
  chainingKey: Uint8Array;
  handshakeHash: Uint8Array;
}

export interface SessionKeys {
  sendCipher: CipherState;
  recvCipher: CipherState;
  handshakeHash: Uint8Array;
}

// Generate X25519 keypair from random or from Ed25519 private key
export function generateX25519KeyPair(): KeyPair {
  const privateKey = randomBytes(32);
  const publicKey = x25519.getPublicKey(privateKey);
  return { publicKey, privateKey };
}

// Convert Ed25519 private key to X25519 (for use with Noise)
export function ed25519PrivateToX25519(ed25519Private: Uint8Array): Uint8Array {
  // Ed25519 private key is hashed, take first 32 bytes for X25519
  const hash = sha256(ed25519Private);
  // Clamp for X25519
  hash[0] &= 248;
  hash[31] &= 127;
  hash[31] |= 64;
  return hash;
}

// HKDF-based key derivation
function hkdfExpand(chainingKey: Uint8Array, input: Uint8Array): [Uint8Array, Uint8Array] {
  const output = hkdf(sha256, input, chainingKey, undefined, 64);
  return [output.slice(0, 32), output.slice(32, 64)];
}

// Mix key material into handshake state
function mixKey(state: HandshakeState, inputKeyMaterial: Uint8Array): void {
  const [newCk, tempK] = hkdfExpand(state.chainingKey, inputKeyMaterial);
  state.chainingKey = newCk;
  // temp key is used for encryption during handshake (not implemented here for simplicity)
}

// Mix hash for transcript binding
function mixHash(state: HandshakeState, data: Uint8Array): void {
  const combined = new Uint8Array(state.handshakeHash.length + data.length);
  combined.set(state.handshakeHash);
  combined.set(data, state.handshakeHash.length);
  state.handshakeHash = sha256(combined);
}

// Initialize handshake state
export function initHandshake(localStatic: KeyPair, initiator: boolean): HandshakeState {
  const protocolNameBytes = new TextEncoder().encode(PROTOCOL_NAME);
  const initialHash = sha256(protocolNameBytes);

  return {
    localEphemeral: generateX25519KeyPair(),
    localStatic,
    chainingKey: initialHash,
    handshakeHash: initialHash,
  };
}

// Perform X25519 DH
function dh(privateKey: Uint8Array, publicKey: Uint8Array): Uint8Array {
  return x25519.getSharedSecret(privateKey, publicKey);
}

// Initiator handshake message 1: -> e
export function initiatorMsg1(state: HandshakeState): Uint8Array {
  mixHash(state, state.localEphemeral.publicKey);
  return state.localEphemeral.publicKey;
}

// Responder processes message 1 and creates message 2: <- e, ee, s, es
export function responderMsg2(state: HandshakeState, msg1: Uint8Array): Uint8Array {
  // Process initiator's ephemeral
  state.remoteEphemeralPub = msg1;
  mixHash(state, msg1);

  // Send our ephemeral
  mixHash(state, state.localEphemeral.publicKey);

  // DH: ee
  const ee = dh(state.localEphemeral.privateKey, state.remoteEphemeralPub);
  mixKey(state, ee);

  // DH: es (our static, their ephemeral)
  const es = dh(state.localStatic.privateKey, state.remoteEphemeralPub);
  mixKey(state, es);

  // Send our static (in real Noise, this would be encrypted)
  mixHash(state, state.localStatic.publicKey);

  // Combine ephemeral + static for response
  const response = new Uint8Array(64);
  response.set(state.localEphemeral.publicKey, 0);
  response.set(state.localStatic.publicKey, 32);

  return response;
}

// Initiator processes message 2 and creates message 3: -> s, se
export function initiatorMsg3(state: HandshakeState, msg2: Uint8Array): Uint8Array {
  // Extract responder's ephemeral and static
  state.remoteEphemeralPub = msg2.slice(0, 32);
  state.remoteStaticPub = msg2.slice(32, 64);

  mixHash(state, state.remoteEphemeralPub);

  // DH: ee
  const ee = dh(state.localEphemeral.privateKey, state.remoteEphemeralPub);
  mixKey(state, ee);

  // DH: es (their static, our ephemeral)
  const es = dh(state.localEphemeral.privateKey, state.remoteStaticPub);
  mixKey(state, es);

  mixHash(state, state.remoteStaticPub);

  // Send our static
  mixHash(state, state.localStatic.publicKey);

  // DH: se (our static, their ephemeral)
  const se = dh(state.localStatic.privateKey, state.remoteEphemeralPub);
  mixKey(state, se);

  return state.localStatic.publicKey;
}

// Responder processes message 3 and finalizes
export function responderFinalize(state: HandshakeState, msg3: Uint8Array): void {
  state.remoteStaticPub = msg3;
  mixHash(state, msg3);

  // DH: se (their static, our ephemeral)
  const se = dh(state.localEphemeral.privateKey, state.remoteStaticPub);
  mixKey(state, se);
}

// Derive session keys after handshake
export function deriveSessionKeys(state: HandshakeState, initiator: boolean): SessionKeys {
  const [k1, k2] = hkdfExpand(state.chainingKey, new Uint8Array(0));

  return {
    sendCipher: { key: initiator ? k1 : k2, nonce: 0n },
    recvCipher: { key: initiator ? k2 : k1, nonce: 0n },
    handshakeHash: state.handshakeHash,
  };
}

// Encrypt with ChaCha20-Poly1305
export function encrypt(cipher: CipherState, plaintext: Uint8Array, aad?: Uint8Array): Uint8Array {
  const nonce = new Uint8Array(12);
  const view = new DataView(nonce.buffer);
  view.setBigUint64(4, cipher.nonce, true); // little-endian, last 8 bytes

  const chacha = chacha20poly1305(cipher.key, nonce, aad);
  const ciphertext = chacha.encrypt(plaintext);

  cipher.nonce++;

  return ciphertext;
}

// Decrypt with ChaCha20-Poly1305
export function decrypt(cipher: CipherState, ciphertext: Uint8Array, aad?: Uint8Array): Uint8Array {
  const nonce = new Uint8Array(12);
  const view = new DataView(nonce.buffer);
  view.setBigUint64(4, cipher.nonce, true);

  const chacha = chacha20poly1305(cipher.key, nonce, aad);
  const plaintext = chacha.decrypt(ciphertext);

  cipher.nonce++;

  return plaintext;
}
