/**
 * Gateway CLI Commands
 *
 * Commands for managing gateway mode:
 * - gateway init: Initialize gateway mode
 * - gateway identity list: List loaded identities
 * - gateway identity add: Add identity to config
 * - gateway identity remove: Remove identity from config
 */

import { Command } from 'commander';
import * as fs from 'fs';
import * as path from 'path';
import { getDataDir, createIdentity, saveIdentity } from '../identity/keys.js';
import {
  isGatewayMode,
  loadGatewayConfig,
  saveGatewayConfig,
  createInitialGatewayConfig,
} from '../daemon/gateway-config.js';
import type { GatewayConfig, IdentityConfig } from '../types/gateway.js';
import * as readline from 'readline';

/**
 * Prompt user for input
 */
function prompt(question: string): Promise<string> {
  const rl = readline.createInterface({
    input: process.stdin,
    output: process.stdout,
  });

  return new Promise((resolve) => {
    rl.question(question, (answer) => {
      rl.close();
      resolve(answer);
    });
  });
}

/**
 * Prompt for password (hidden input)
 */
function promptPassword(question: string): Promise<string> {
  return new Promise((resolve) => {
    const rl = readline.createInterface({
      input: process.stdin,
      output: process.stdout,
    });

    process.stdin.on('data', (char) => {
      const byte = char.toString();
      if (byte === '\n' || byte === '\r' || byte === '\u0004') {
        process.stdin.pause();
      } else if (byte === '\u0003') {
        process.exit(1);
      }
    });

    rl.question(question, (answer) => {
      rl.close();
      console.log(); // New line after password
      resolve(answer);
    });

    (rl as any)._writeToOutput = () => {}; // Hide input
  });
}

/**
 * Gateway init command
 */
async function gatewayInit(options: {
  port?: string;
  nick?: string;
  migrate?: boolean;
  password?: string;
  testnet?: boolean;
}): Promise<void> {
  const dataDir = getDataDir();

  // Check if gateway mode already exists
  if (isGatewayMode()) {
    console.error('Gateway mode already initialized.');
    console.error('Gateway config exists at:', path.join(dataDir, 'gateway-config.json'));
    process.exit(1);
  }

  console.log('Initializing gateway mode...\n');

  // Check for existing legacy identity
  const legacyIdentityPath = path.join(dataDir, 'identity.enc');
  const hasLegacyIdentity = fs.existsSync(legacyIdentityPath);

  let principal: string;
  let nick: string | undefined;

  if (hasLegacyIdentity && options.migrate !== false) {
    // Offer to migrate existing identity
    const migrate = await prompt(
      'Found existing identity. Migrate to gateway mode? (y/n): '
    );

    if (migrate.toLowerCase() === 'y') {
      // Prompt for password to decrypt existing identity
      const password = await promptPassword('Enter password to decrypt identity: ');

      // Load existing identity
      const { loadIdentity } = await import('../identity/keys.js');
      const identity = loadIdentity(password);

      if (!identity) {
        console.error('Failed to load existing identity. Invalid password?');
        process.exit(1);
      }

      principal = identity.principal;
      nick = options.nick || identity.nick;

      console.log(`\nMigrating identity: ${principal}`);

      // Create per-identity directory
      const identityDir = path.join(dataDir, 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      // Move identity.enc to per-identity directory
      const newIdentityPath = path.join(identityDir, 'identity.enc');
      fs.copyFileSync(legacyIdentityPath, newIdentityPath);

      // Move inbox, outbox, peers if they exist
      const files = ['inbox.json', 'outbox.json', 'peers.json'];
      for (const file of files) {
        const oldPath = path.join(dataDir, file);
        if (fs.existsSync(oldPath)) {
          const newPath = path.join(identityDir, file);
          fs.copyFileSync(oldPath, newPath);
        }
      }

      console.log(`✓ Migrated identity to: ${identityDir}`);
    } else {
      // Create new identity
      const result = await createNewIdentity(options.nick, options.password, options.testnet);
      principal = result.principal;
      nick = result.nick;
    }
  } else {
    // Create new identity
    const result = await createNewIdentity(options.nick, options.password, options.testnet);
    principal = result.principal;
    nick = result.nick;
  }

  // Create gateway config
  const p2pPort = options.port ? parseInt(options.port, 10) : 9000;
  const config = createInitialGatewayConfig(principal, nick, p2pPort);

  saveGatewayConfig(config);

  console.log('\n✓ Gateway mode initialized successfully!');
  console.log(`\nConfiguration saved to: ${path.join(dataDir, 'gateway-config.json')}`);
  console.log(`Identity: ${principal}${nick ? ` (${nick})` : ''}`);
  console.log(`P2P Port: ${p2pPort}`);
  console.log('\nYou can now start the daemon with: clawchat daemon start');
}

/**
 * Create a new identity
 */
async function createNewIdentity(
  nickname?: string,
  passwordArg?: string,
  testnetArg?: boolean
): Promise<{ principal: string; nick?: string }> {
  console.log('\nCreating new identity...');

  // Ask for testnet or mainnet (unless provided via CLI)
  let testnet: boolean;
  if (testnetArg !== undefined) {
    testnet = testnetArg;
  } else {
    const networkInput = await prompt('Use testnet? (Y/n): ');
    testnet = networkInput.toLowerCase() !== 'n';
  }

  // Create identity
  const identity = await createIdentity(testnet);

  console.log('\n✓ Identity created successfully!');
  console.log(`Principal: ${identity.principal}`);
  console.log(`\n⚠️  IMPORTANT: Back up your seed phrase!`);
  console.log(`Seed phrase: ${identity.mnemonic}\n`);

  // Use provided password or prompt for one
  let password1: string;
  if (passwordArg) {
    password1 = passwordArg;
  } else {
    password1 = await promptPassword('Enter password to encrypt identity (min 12 chars): ');
  }

  if (password1.length < 12) {
    console.error('Password must be at least 12 characters');
    process.exit(1);
  }

  // Skip confirmation if password provided via CLI
  if (!passwordArg) {
    const password2 = await promptPassword('Confirm password: ');

    if (password1 !== password2) {
      console.error('Passwords do not match');
      process.exit(1);
    }
  }

  // Ask for nickname
  let nick = nickname;
  if (!nick) {
    const nickInput = await prompt('Nickname (optional, press enter to skip): ');
    nick = nickInput.trim() || undefined;
  }

  if (nick) {
    identity.nick = nick;
  }

  // Create per-identity directory
  const dataDir = getDataDir();
  const identityDir = path.join(dataDir, 'identities', identity.principal);
  fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

  // Save identity
  process.env.CLAWCHAT_DATA_DIR = identityDir;
  try {
    saveIdentity(identity, password1);
    console.log(`\n✓ Identity saved to: ${identityDir}`);
  } finally {
    delete process.env.CLAWCHAT_DATA_DIR;
  }

  // Create empty inbox, outbox, peers
  fs.writeFileSync(path.join(identityDir, 'inbox.json'), '[]', { mode: 0o600 });
  fs.writeFileSync(path.join(identityDir, 'outbox.json'), '[]', { mode: 0o600 });
  fs.writeFileSync(path.join(identityDir, 'peers.json'), '[]', { mode: 0o600 });

  return { principal: identity.principal, nick };
}

/**
 * Gateway identity list command
 */
function gatewayIdentityList(): void {
  if (!isGatewayMode()) {
    console.error('Gateway mode not initialized. Run "clawchat gateway init" first.');
    process.exit(1);
  }

  const config = loadGatewayConfig();

  console.log(`\nGateway Identities (${config.identities.length}):\n`);

  for (const identity of config.identities) {
    console.log(`  ${identity.principal}`);
    if (identity.nick) {
      console.log(`    Nickname: ${identity.nick}`);
    }
    console.log(`    Autoload: ${identity.autoload}`);
    console.log(`    Allow Local: ${identity.allowLocal}`);
    console.log(
      `    Allowed Peers: ${identity.allowedRemotePeers.length === 1 && identity.allowedRemotePeers[0] === '*' ? 'All (*)' : identity.allowedRemotePeers.join(', ')}`
    );
    console.log(`    OpenClaw Wake: ${identity.openclawWake}`);
    console.log();
  }
}

/**
 * Gateway identity add command
 */
async function gatewayIdentityAdd(options: {
  principal?: string;
  nick?: string;
  noAutoload?: boolean;
  allowPeers?: string;
}): Promise<void> {
  if (!isGatewayMode()) {
    console.error('Gateway mode not initialized. Run "clawchat gateway init" first.');
    process.exit(1);
  }

  const config = loadGatewayConfig();

  // Get principal
  let principal = options.principal;
  if (!principal) {
    // Create new identity
    const result = await createNewIdentity(options.nick);
    principal = result.principal;
    options.nick = result.nick;
  }

  // Check if already exists
  if (config.identities.some((id) => id.principal === principal)) {
    console.error(`Identity ${principal} already exists in config`);
    process.exit(1);
  }

  // Parse allowed peers
  let allowedRemotePeers: string[] = ['*'];
  if (options.allowPeers) {
    allowedRemotePeers = options.allowPeers.split(',').map((p) => p.trim());
  }

  // Create identity config
  const identityConfig: IdentityConfig = {
    principal,
    nick: options.nick,
    autoload: !options.noAutoload,
    allowLocal: true,
    allowedRemotePeers,
    openclawWake: true,
  };

  // Add to config
  config.identities.push(identityConfig);
  saveGatewayConfig(config);

  console.log(`\n✓ Identity added: ${principal}${options.nick ? ` (${options.nick})` : ''}`);
}

/**
 * Gateway identity remove command
 */
function gatewayIdentityRemove(principalOrNick: string): void {
  if (!isGatewayMode()) {
    console.error('Gateway mode not initialized. Run "clawchat gateway init" first.');
    process.exit(1);
  }

  const config = loadGatewayConfig();

  // Find identity by principal or nickname
  const index = config.identities.findIndex(
    (id) => id.principal === principalOrNick || id.nick === principalOrNick
  );

  if (index === -1) {
    console.error(`Identity not found: ${principalOrNick}`);
    process.exit(1);
  }

  const removed = config.identities[index];
  config.identities.splice(index, 1);

  if (config.identities.length === 0) {
    console.error('Cannot remove last identity. At least one identity is required.');
    process.exit(1);
  }

  saveGatewayConfig(config);

  console.log(`\n✓ Identity removed: ${removed.principal}${removed.nick ? ` (${removed.nick})` : ''}`);
  console.log('\nNote: Identity files are not deleted, only removed from config.');
}

/**
 * Register gateway commands
 */
export function registerGatewayCommands(program: Command): void {
  const gateway = program.command('gateway').description('Manage gateway mode');

  // gateway init
  gateway
    .command('init')
    .description('Initialize gateway mode')
    .option('-p, --port <port>', 'P2P port', '9000')
    .option('-n, --nick <nickname>', 'Nickname for identity')
    .option('--no-migrate', 'Skip migration of existing identity')
    .option('--password <password>', 'Password for identity encryption (min 12 chars)')
    .option('--testnet', 'Use testnet (default)')
    .option('--mainnet', 'Use mainnet')
    .action((options) => {
      // Convert --mainnet to testnet=false
      if (options.mainnet) {
        options.testnet = false;
      } else if (options.testnet === undefined && !options.mainnet) {
        // Will prompt interactively
      } else {
        options.testnet = true;
      }
      return gatewayInit(options);
    });

  // gateway identity commands
  const identity = gateway.command('identity').description('Manage identities');

  identity
    .command('list')
    .description('List configured identities')
    .action(gatewayIdentityList);

  identity
    .command('add')
    .description('Add identity to gateway')
    .option('--principal <principal>', 'Identity principal (if already exists)')
    .option('-n, --nick <nickname>', 'Nickname for identity')
    .option('--no-autoload', 'Do not autoload this identity on daemon start')
    .option('--allow-peers <peers>', 'Comma-separated list of allowed peer principals (* for all)')
    .action(gatewayIdentityAdd);

  identity
    .command('remove <principal-or-nick>')
    .description('Remove identity from gateway config')
    .action(gatewayIdentityRemove);
}
