import { describe, it, expect } from 'vitest';
import {
  frameMessage,
  unframeMessage,
  encodeMessage,
  decodeMessage,
  createHelloMessage,
  createAuthMessage,
  createAuthOkMessage,
  createPingMessage,
  createPongMessage,
  encodeChatMessage,
  decodeChatMessage,
  type ChatMessage,
} from '../wire/framing.js';
import { MessageType } from '../types.js';
import { randomBytes } from '@noble/hashes/utils';

describe('Wire Protocol', () => {
  describe('frameMessage / unframeMessage', () => {
    it('frames message with length prefix', () => {
      const data = new Uint8Array([1, 2, 3, 4, 5]);
      const framed = frameMessage(data);

      expect(framed.length).toBe(4 + 5); // 4-byte length + data
      expect(framed[0]).toBe(0);
      expect(framed[1]).toBe(0);
      expect(framed[2]).toBe(0);
      expect(framed[3]).toBe(5); // length in big-endian
    });

    it('unframes message correctly', () => {
      const original = new Uint8Array([1, 2, 3, 4, 5]);
      const framed = frameMessage(original);
      const result = unframeMessage(framed);

      expect(result).not.toBeNull();
      expect(Buffer.from(result!.data).toString('hex'))
        .toBe(Buffer.from(original).toString('hex'));
      expect(result!.remaining.length).toBe(0);
    });

    it('handles partial frames', () => {
      const data = new Uint8Array([1, 2, 3, 4, 5]);
      const framed = frameMessage(data);

      // Only send first 3 bytes (not enough for length)
      const partial1 = framed.slice(0, 3);
      expect(unframeMessage(partial1)).toBeNull();

      // Send length but not full data
      const partial2 = framed.slice(0, 6);
      expect(unframeMessage(partial2)).toBeNull();
    });

    it('handles multiple frames in buffer', () => {
      const data1 = new Uint8Array([1, 2, 3]);
      const data2 = new Uint8Array([4, 5, 6, 7]);

      const framed1 = frameMessage(data1);
      const framed2 = frameMessage(data2);

      const combined = new Uint8Array(framed1.length + framed2.length);
      combined.set(framed1);
      combined.set(framed2, framed1.length);

      const result1 = unframeMessage(combined);
      expect(result1).not.toBeNull();
      expect(result1!.data).toEqual(data1);

      const result2 = unframeMessage(result1!.remaining);
      expect(result2).not.toBeNull();
      expect(result2!.data).toEqual(data2);
    });

    it('rejects oversized messages', () => {
      const huge = new Uint8Array(20 * 1024 * 1024); // 20 MB
      expect(() => frameMessage(huge)).toThrow('Message too large');
    });
  });

  describe('encodeMessage / decodeMessage', () => {
    it('encodes and decodes HELLO message', () => {
      const nodePublicKey = randomBytes(32);
      const nonce = randomBytes(16);
      const original = createHelloMessage(nodePublicKey, nonce);

      const encoded = encodeMessage(original);
      const decoded = decodeMessage(encoded);

      expect(decoded.type).toBe(MessageType.HELLO);
      expect(decoded.version).toBe(1);
      expect(Buffer.from(decoded.nodePublicKey as Uint8Array).toString('hex'))
        .toBe(Buffer.from(nodePublicKey).toString('hex'));
    });

    it('encodes and decodes AUTH message', () => {
      const principal = 'stacks:ST1234test';
      const attestation = { version: 1, data: 'test' };
      const sessionNonce = randomBytes(32);
      const original = createAuthMessage(principal, attestation, sessionNonce);

      const encoded = encodeMessage(original);
      const decoded = decodeMessage(encoded);

      expect(decoded.type).toBe(MessageType.AUTH);
      expect(decoded.principal).toBe(principal);
    });

    it('encodes and decodes AUTH_OK message', () => {
      const original = createAuthOkMessage();

      const encoded = encodeMessage(original);
      const decoded = decodeMessage(encoded);

      expect(decoded.type).toBe(MessageType.AUTH_OK);
    });

    it('encodes and decodes PING/PONG messages', () => {
      const nonce = randomBytes(16);

      const ping = createPingMessage(nonce);
      const encodedPing = encodeMessage(ping);
      const decodedPing = decodeMessage(encodedPing);
      expect(decodedPing.type).toBe(MessageType.PING);

      const pong = createPongMessage(nonce);
      const encodedPong = encodeMessage(pong);
      const decodedPong = decodeMessage(encodedPong);
      expect(decodedPong.type).toBe(MessageType.PONG);
    });
  });

  describe('Chat Messages', () => {
    it('encodes and decodes chat message', () => {
      const original: ChatMessage = {
        id: 'msg-123',
        from: 'stacks:ST1234',
        content: 'Hello, world!',
        timestamp: Date.now(),
      };

      const encoded = encodeChatMessage(original);
      const decoded = decodeChatMessage(encoded);

      expect(decoded.id).toBe(original.id);
      expect(decoded.from).toBe(original.from);
      expect(decoded.content).toBe(original.content);
      expect(decoded.timestamp).toBe(original.timestamp);
    });

    it('handles unicode content', () => {
      const original: ChatMessage = {
        id: 'msg-unicode',
        from: 'stacks:ST1234',
        content: '你好世界 🌍 مرحبا',
        timestamp: Date.now(),
      };

      const encoded = encodeChatMessage(original);
      const decoded = decodeChatMessage(encoded);

      expect(decoded.content).toBe(original.content);
    });

    it('handles long messages', () => {
      const longContent = 'A'.repeat(10000);
      const original: ChatMessage = {
        id: 'msg-long',
        from: 'stacks:ST1234',
        content: longContent,
        timestamp: Date.now(),
      };

      const encoded = encodeChatMessage(original);
      const decoded = decodeChatMessage(encoded);

      expect(decoded.content).toBe(longContent);
      expect(decoded.content.length).toBe(10000);
    });
  });
});
