/**
 * Multi-Identity SNaP2P Tests (Phase 3)
 *
 * Tests for SNaP2P protocol with multiple identities and session binding
 */

import { describe, it, expect, beforeEach } from 'vitest';
import type { FullIdentity } from '../identity/keys.js';
import type { IdentityResolver } from '../net/snap2p-protocol.js';

describe('Multi-Identity SNaP2P', () => {
  // Mock identities for testing
  const mockIdentity1: FullIdentity = {
    principal: 'stacks:ST1ALICE',
    address: 'ST1ALICE',
    publicKey: new Uint8Array(32),
    privateKey: new Uint8Array(32),
    mnemonic: 'test mnemonic 1',
    walletPublicKeyHex: 'aabbcc',
    walletPrivateKeyHex: 'ddeeff',
    testnet: true,
  };

  const mockIdentity2: FullIdentity = {
    principal: 'stacks:ST2BOB',
    address: 'ST2BOB',
    publicKey: new Uint8Array(32),
    privateKey: new Uint8Array(32),
    mnemonic: 'test mnemonic 2',
    walletPublicKeyHex: '112233',
    walletPrivateKeyHex: '445566',
    testnet: true,
  };

  describe('IdentityResolver', () => {
    it('returns correct identity based on remote principal', () => {
      // Create a resolver that maps specific peers to specific identities
      const resolver: IdentityResolver = (remotePrincipal?: string) => {
        if (!remotePrincipal) {
          // No remote peer specified, return default
          return mockIdentity1;
        }

        // Map specific peers to specific identities
        if (remotePrincipal === 'stacks:ST3CHARLIE') {
          return mockIdentity1; // Alice handles Charlie
        } else if (remotePrincipal === 'stacks:ST4DAVE') {
          return mockIdentity2; // Bob handles Dave
        }

        return null; // Not allowed
      };

      // Test default (no remote principal)
      expect(resolver()).toEqual(mockIdentity1);

      // Test specific peer routing
      expect(resolver('stacks:ST3CHARLIE')).toEqual(mockIdentity1);
      expect(resolver('stacks:ST4DAVE')).toEqual(mockIdentity2);

      // Test rejected peer
      expect(resolver('stacks:ST5UNKNOWN')).toBeNull();
    });

    it('handles wildcard ACL (allow all)', () => {
      // Resolver that allows all peers (ACL with "*")
      const resolver: IdentityResolver = (remotePrincipal?: string) => {
        if (!remotePrincipal) {
          return mockIdentity1;
        }
        // Allow all peers to connect to identity1
        return mockIdentity1;
      };

      // Any peer should get identity1
      expect(resolver('stacks:ST3ANYONE')).toEqual(mockIdentity1);
      expect(resolver('stacks:ST4ANYONE')).toEqual(mockIdentity1);
    });

    it('handles multiple identities allowing same peer', () => {
      // Resolver where multiple identities allow the same peer
      // Return the first matching identity
      const resolver: IdentityResolver = (remotePrincipal?: string) => {
        if (!remotePrincipal) {
          return mockIdentity1;
        }

        // Both identities allow this peer, return first one
        if (remotePrincipal === 'stacks:ST3TRUSTED') {
          return mockIdentity1; // Could be either, choose first
        }

        return null;
      };

      expect(resolver('stacks:ST3TRUSTED')).toEqual(mockIdentity1);
    });

    it('rejects peer not allowed by any identity', () => {
      // Resolver with strict ACL
      const resolver: IdentityResolver = (remotePrincipal?: string) => {
        if (!remotePrincipal) {
          return mockIdentity1;
        }

        // Only specific peers allowed
        const allowedPeers = ['stacks:ST3ALICE', 'stacks:ST4BOB'];
        if (allowedPeers.includes(remotePrincipal)) {
          return mockIdentity1;
        }

        return null;
      };

      // Allowed peer
      expect(resolver('stacks:ST3ALICE')).toEqual(mockIdentity1);

      // Rejected peer
      expect(resolver('stacks:ST5UNKNOWN')).toBeNull();
    });
  });

  describe('Session Identity Binding', () => {
    it('session maintains reference to local identity', () => {
      // This is tested via the localPrincipal getter added to Snap2pSession
      // The getter returns the identity.principal for the session

      // In actual usage:
      // const session = new Snap2pSession(stream, mockIdentity1, peerId);
      // expect(session.localPrincipal).toBe('stacks:ST1ALICE');

      // We can't create an actual Snap2pSession in unit tests without a real stream
      // This is verified in integration tests
      expect(mockIdentity1.principal).toBe('stacks:ST1ALICE');
      expect(mockIdentity2.principal).toBe('stacks:ST2BOB');
    });
  });

  describe('Gateway Mode Identity Selection', () => {
    it('selects correct identity for incoming connection', () => {
      // Simulate gateway mode identity selection logic
      const allowedPeersByIdentity = new Map<string, string[]>([
        ['stacks:ST1ALICE', ['*']], // Alice allows all
        ['stacks:ST2BOB', ['stacks:ST3CHARLIE']], // Bob only allows Charlie
      ]);

      function selectIdentityForPeer(remotePrincipal: string): FullIdentity | null {
        // Check which identities allow this peer
        for (const [identityPrincipal, allowedPeers] of allowedPeersByIdentity) {
          if (allowedPeers.includes('*') || allowedPeers.includes(remotePrincipal)) {
            // Return the first matching identity
            return identityPrincipal === 'stacks:ST1ALICE' ? mockIdentity1 : mockIdentity2;
          }
        }
        return null;
      }

      // Alice allows all, so any peer should get Alice
      expect(selectIdentityForPeer('stacks:ST5RANDOM')?.principal).toBe('stacks:ST1ALICE');

      // Charlie can connect to Bob specifically
      const identity = selectIdentityForPeer('stacks:ST3CHARLIE');
      expect(identity).not.toBeNull();
      expect(['stacks:ST1ALICE', 'stacks:ST2BOB']).toContain(identity!.principal);
    });

    it('uses first identity as default for outgoing connections', () => {
      // When making an outgoing connection without specifying identity,
      // use the first loaded identity
      const identities = [mockIdentity1, mockIdentity2];
      const defaultIdentity = identities[0];

      expect(defaultIdentity.principal).toBe('stacks:ST1ALICE');
    });

    it('allows explicit identity selection for outgoing connections', () => {
      // When making an outgoing connection, can specify which identity to use
      const identities = [mockIdentity1, mockIdentity2];

      function getIdentityByPrincipal(principal: string): FullIdentity | null {
        return identities.find((id) => id.principal === principal) || null;
      }

      // Request specific identity for outgoing connection
      const identity = getIdentityByPrincipal('stacks:ST2BOB');
      expect(identity).toEqual(mockIdentity2);
    });
  });

  describe('ACL Enforcement', () => {
    it('rejects connection when no identity allows peer', () => {
      // Simulate strict ACL where no identity allows a peer
      const allowedPeersByIdentity = new Map<string, string[]>([
        ['stacks:ST1ALICE', ['stacks:ST3CHARLIE']],
        ['stacks:ST2BOB', ['stacks:ST4DAVE']],
      ]);

      function isPeerAllowed(remotePrincipal: string): boolean {
        for (const allowedPeers of allowedPeersByIdentity.values()) {
          if (allowedPeers.includes('*') || allowedPeers.includes(remotePrincipal)) {
            return true;
          }
        }
        return false;
      }

      // Allowed peers
      expect(isPeerAllowed('stacks:ST3CHARLIE')).toBe(true);
      expect(isPeerAllowed('stacks:ST4DAVE')).toBe(true);

      // Rejected peer
      expect(isPeerAllowed('stacks:ST5UNKNOWN')).toBe(false);
    });

    it('allows connection with wildcard ACL', () => {
      const allowedPeersByIdentity = new Map<string, string[]>([
        ['stacks:ST1ALICE', ['*']], // Alice allows all
      ]);

      function isPeerAllowed(remotePrincipal: string): boolean {
        for (const allowedPeers of allowedPeersByIdentity.values()) {
          if (allowedPeers.includes('*')) {
            return true;
          }
        }
        return false;
      }

      // Any peer should be allowed
      expect(isPeerAllowed('stacks:ST3ANYONE')).toBe(true);
      expect(isPeerAllowed('stacks:ST4ANYONE')).toBe(true);
    });
  });

  describe('Mode Detection', () => {
    it('detects legacy mode when single identity provided', () => {
      // In legacy mode, identity is a FullIdentity object
      const identity: FullIdentity = mockIdentity1;
      const isGatewayMode = typeof identity === 'function';

      expect(isGatewayMode).toBe(false);
    });

    it('detects gateway mode when resolver provided', () => {
      // In gateway mode, identity is a function (resolver)
      const resolver: IdentityResolver = () => mockIdentity1;
      const isGatewayMode = typeof resolver === 'function';

      expect(isGatewayMode).toBe(true);
    });
  });
});
