/**
 * Message Routing Tests (Phase 2)
 *
 * Tests for MessageRouter ACL enforcement and message routing
 */

import { describe, it, expect, beforeEach } from 'vitest';
import { MessageRouter } from '../daemon/message-router.js';
import { IdentityManager } from '../daemon/identity-manager.js';
import type { Message } from '../types.js';
import type { IdentityConfig } from '../types/gateway.js';

describe('MessageRouter', () => {
  let manager: IdentityManager;
  let router: MessageRouter;

  beforeEach(() => {
    manager = new IdentityManager();
    router = new MessageRouter(manager);
  });

  describe('routeInbound', () => {
    it('routes message to correct identity', () => {
      // Create a mock loaded identity
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1RECEIVER',
          address: 'ST1RECEIVER',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1RECEIVER',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      // Mock getIdentity to return our mock
      const originalGet = manager.getIdentity.bind(manager);
      manager.getIdentity = (principal: string) => {
        if (principal === 'stacks:ST1RECEIVER') {
          return mockIdentity;
        }
        return originalGet(principal);
      };

      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1SENDER',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeInbound(message, 'stacks:ST1SENDER');

      expect(result.success).toBe(true);
      expect(result.identity).toBeDefined();
      expect(result.identity!.identity.principal).toBe('stacks:ST1RECEIVER');
    });

    it('rejects message when identity not found', () => {
      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1SENDER',
        to: 'stacks:ST1NOTFOUND',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeInbound(message, 'stacks:ST1SENDER');

      expect(result.success).toBe(false);
      expect(result.error).toContain('No loaded identity found');
    });

    it('enforces ACL with wildcard allowed', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1RECEIVER',
          address: 'ST1RECEIVER',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1RECEIVER',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'], // Allow all
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = (principal: string) => {
        if (principal === 'stacks:ST1RECEIVER') {
          return mockIdentity;
        }
        return null;
      };

      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1ANYONE',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeInbound(message, 'stacks:ST1ANYONE');

      expect(result.success).toBe(true);
    });

    it('enforces ACL with specific peer allowed', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1RECEIVER',
          address: 'ST1RECEIVER',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1RECEIVER',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['stacks:ST1ALLOWED'], // Only one peer
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = (principal: string) => {
        if (principal === 'stacks:ST1RECEIVER') {
          return mockIdentity;
        }
        return null;
      };

      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1ALLOWED',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeInbound(message, 'stacks:ST1ALLOWED');

      expect(result.success).toBe(true);
    });

    it('rejects message from non-allowed peer', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1RECEIVER',
          address: 'ST1RECEIVER',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1RECEIVER',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['stacks:ST1ALLOWED'], // Only one peer
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = (principal: string) => {
        if (principal === 'stacks:ST1RECEIVER') {
          return mockIdentity;
        }
        return null;
      };

      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1NOTALLOWED',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeInbound(message, 'stacks:ST1NOTALLOWED');

      expect(result.success).toBe(false);
      expect(result.error).toContain('not in allowedRemotePeers');
    });
  });

  describe('routeOutbound', () => {
    it('uses message.from to find source identity', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1SENDER',
          address: 'ST1SENDER',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1SENDER',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = (principal: string) => {
        if (principal === 'stacks:ST1SENDER') {
          return mockIdentity;
        }
        return null;
      };

      const message: Message = {
        id: 'msg1',
        from: 'stacks:ST1SENDER',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeOutbound(message);

      expect(result.success).toBe(true);
      expect(result.identity).toBeDefined();
      expect(result.identity!.identity.principal).toBe('stacks:ST1SENDER');
    });

    it('uses default identity when message.from not set', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1DEFAULT',
          address: 'ST1DEFAULT',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1DEFAULT',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getAllIdentities = () => [mockIdentity];

      const message: Message = {
        id: 'msg1',
        from: '',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeOutbound(message);

      expect(result.success).toBe(true);
      expect(result.identity).toBeDefined();
      expect(result.identity!.identity.principal).toBe('stacks:ST1DEFAULT');
    });

    it('fails when no identities loaded', () => {
      manager.getAllIdentities = () => [];

      const message: Message = {
        id: 'msg1',
        from: '',
        to: 'stacks:ST1RECEIVER',
        content: 'Hello',
        timestamp: Date.now(),
        status: 'pending',
      };

      const result = router.routeOutbound(message);

      expect(result.success).toBe(false);
      expect(result.error).toContain('No identities loaded');
    });
  });

  describe('isLocalAccessAllowed', () => {
    it('returns true when allowLocal is true', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1TEST',
          address: 'ST1TEST',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1TEST',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = () => mockIdentity;

      expect(router.isLocalAccessAllowed('stacks:ST1TEST')).toBe(true);
    });

    it('returns false when allowLocal is false', () => {
      const mockIdentity = {
        identity: {
          principal: 'stacks:ST1TEST',
          address: 'ST1TEST',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1TEST',
          autoload: true,
          allowLocal: false,
          allowedRemotePeers: ['*'],
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getIdentity = () => mockIdentity;

      expect(router.isLocalAccessAllowed('stacks:ST1TEST')).toBe(false);
    });

    it('returns false when identity not found', () => {
      manager.getIdentity = () => null;

      expect(router.isLocalAccessAllowed('stacks:ST1NOTFOUND')).toBe(false);
    });
  });

  describe('findIdentitiesForPeer', () => {
    it('returns identities that allow the peer', () => {
      const identity1 = {
        identity: {
          principal: 'stacks:ST1IDENTITY1',
          address: 'ST1IDENTITY1',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1IDENTITY1',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'], // Allows all
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test1',
      };

      const identity2 = {
        identity: {
          principal: 'stacks:ST1IDENTITY2',
          address: 'ST1IDENTITY2',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1IDENTITY2',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['stacks:ST1SPECIFICPEER'], // Only specific peer
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test2',
      };

      manager.getAllIdentities = () => [identity1, identity2];

      // Check which identities allow a random peer
      const results = router.findIdentitiesForPeer('stacks:ST1RANDOMPEER');

      expect(results.length).toBe(1);
      expect(results[0].identity.principal).toBe('stacks:ST1IDENTITY1');

      // Check which identities allow the specific peer
      const results2 = router.findIdentitiesForPeer('stacks:ST1SPECIFICPEER');

      expect(results2.length).toBe(2);
    });

    it('returns empty array when no identities allow the peer', () => {
      const identity = {
        identity: {
          principal: 'stacks:ST1IDENTITY',
          address: 'ST1IDENTITY',
          publicKey: new Uint8Array(),
          privateKey: new Uint8Array(),
          mnemonic: 'test',
          walletPublicKeyHex: 'test',
          walletPrivateKeyHex: 'test',
          testnet: true,
        },
        config: {
          principal: 'stacks:ST1IDENTITY',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['stacks:ST1ALLOWED'], // Only one specific peer
          openclawWake: false,
        },
        inbox: [],
        outbox: [],
        peers: [],
        dataDir: '/tmp/test',
      };

      manager.getAllIdentities = () => [identity];

      const results = router.findIdentitiesForPeer('stacks:ST1NOTALLOWED');

      expect(results.length).toBe(0);
    });
  });
});
