import { describe, it, expect, beforeEach, afterEach } from 'vitest';
import * as fs from 'fs';
import * as path from 'path';
import {
  createIdentity,
  saveIdentity,
  loadIdentity,
  identityExists,
  generateNodeKeyPair,
  sign,
  verify,
  createAttestation,
  verifyAttestation,
  getDataDir,
  isValidMnemonic,
  recoverIdentity,
} from '../identity/keys.js';

// Use a temp directory for tests
const TEST_DIR = '/tmp/clawchat-test-' + process.pid;
const ORIGINAL_HOME = process.env.HOME;

describe('Identity', () => {
  beforeEach(() => {
    process.env.HOME = TEST_DIR;
    fs.mkdirSync(TEST_DIR, { recursive: true });
  });

  afterEach(() => {
    process.env.HOME = ORIGINAL_HOME;
    fs.rmSync(TEST_DIR, { recursive: true, force: true });
  });

  describe('generateNodeKeyPair', () => {
    it('generates valid Ed25519 keypair', () => {
      const { publicKey, privateKey } = generateNodeKeyPair();

      expect(publicKey).toBeInstanceOf(Uint8Array);
      expect(privateKey).toBeInstanceOf(Uint8Array);
      expect(publicKey.length).toBe(32);
      expect(privateKey.length).toBe(32);
    });

    it('generates unique keypairs', () => {
      const kp1 = generateNodeKeyPair();
      const kp2 = generateNodeKeyPair();

      expect(Buffer.from(kp1.publicKey).toString('hex'))
        .not.toBe(Buffer.from(kp2.publicKey).toString('hex'));
    });
  });

  describe('Stacks wallet', () => {
    it('generates valid mnemonic', async () => {
      const id = await createIdentity(true);
      expect(id.mnemonic).toBeDefined();
      expect(id.mnemonic.split(' ').length).toBe(24);
      expect(isValidMnemonic(id.mnemonic)).toBe(true);
    });

    it('derives testnet address with S prefix', async () => {
      const id = await createIdentity(true);
      expect(id.address).toMatch(/^S[A-Z0-9]+$/);
    });

    it('derives mainnet address with S prefix', async () => {
      const id = await createIdentity(false);
      expect(id.address).toMatch(/^S[A-Z0-9]+$/);
    });

    it('recovers same address from mnemonic', async () => {
      const id1 = await createIdentity(true);
      const id2 = await recoverIdentity(id1.mnemonic, true);

      expect(id2.address).toBe(id1.address);
      expect(id2.principal).toBe(id1.principal);
    });

    it('rejects invalid mnemonic', async () => {
      expect(isValidMnemonic('invalid seed phrase')).toBe(false);
      await expect(recoverIdentity('invalid seed phrase')).rejects.toThrow();
    });
  });

  describe('createIdentity', () => {
    it('creates identity with all required fields', async () => {
      const id = await createIdentity(true);

      expect(id.principal).toMatch(/^stacks:ST/);
      expect(id.address).toMatch(/^ST/);
      expect(id.publicKey).toBeInstanceOf(Uint8Array);
      expect(id.privateKey).toBeInstanceOf(Uint8Array);
      expect(id.principal).toBe(`stacks:${id.address}`);
    });

    it('creates mainnet identity when specified', async () => {
      const id = await createIdentity(false);

      expect(id.principal).toMatch(/^stacks:SP/);
      expect(id.address).toMatch(/^SP/);
    });
  });

  describe('saveIdentity / loadIdentity', () => {
    it('encrypts and decrypts identity with correct password', async () => {
      const original = await createIdentity(true);
      const password = 'test-password-123';

      saveIdentity(original, password);
      const loaded = loadIdentity(password);

      expect(loaded).not.toBeNull();
      expect(loaded!.principal).toBe(original.principal);
      expect(loaded!.address).toBe(original.address);
      expect(Buffer.from(loaded!.publicKey).toString('hex'))
        .toBe(Buffer.from(original.publicKey).toString('hex'));
      expect(Buffer.from(loaded!.privateKey).toString('hex'))
        .toBe(Buffer.from(original.privateKey).toString('hex'));
    });

    it('fails with wrong password', async () => {
      const original = await createIdentity(true);
      saveIdentity(original, 'correct-password');

      expect(() => loadIdentity('wrong-password'))
        .toThrow('Invalid password or corrupted identity file');
    });

    it('returns null when no identity exists', () => {
      const loaded = loadIdentity('any-password');
      expect(loaded).toBeNull();
    });

    it('creates file with restricted permissions', async () => {
      const id = await createIdentity(true);
      saveIdentity(id, 'testpassword12chars');

      const filePath = path.join(getDataDir(), 'identity.enc');
      const stats = fs.statSync(filePath);
      const mode = stats.mode & 0o777;

      expect(mode).toBe(0o600);
    });
  });

  describe('identityExists', () => {
    it('returns false when no identity', () => {
      expect(identityExists()).toBe(false);
    });

    it('returns true after creating identity', async () => {
      const id = await createIdentity(true);
      saveIdentity(id, 'testpassword12chars');

      expect(identityExists()).toBe(true);
    });
  });

  describe('sign / verify', () => {
    it('creates valid signature', () => {
      const { publicKey, privateKey } = generateNodeKeyPair();
      const message = new TextEncoder().encode('Hello, World!');

      const signature = sign(privateKey, message);

      expect(signature).toBeInstanceOf(Uint8Array);
      expect(signature.length).toBe(64);
    });

    it('verifies valid signature', () => {
      const { publicKey, privateKey } = generateNodeKeyPair();
      const message = new TextEncoder().encode('Hello, World!');
      const signature = sign(privateKey, message);

      expect(verify(publicKey, message, signature)).toBe(true);
    });

    it('rejects invalid signature', () => {
      const { publicKey, privateKey } = generateNodeKeyPair();
      const message = new TextEncoder().encode('Hello, World!');
      const signature = sign(privateKey, message);

      // Tamper with signature
      signature[0] ^= 0xff;

      expect(verify(publicKey, message, signature)).toBe(false);
    });

    it('rejects signature from different key', () => {
      const kp1 = generateNodeKeyPair();
      const kp2 = generateNodeKeyPair();
      const message = new TextEncoder().encode('Hello, World!');
      const signature = sign(kp1.privateKey, message);

      expect(verify(kp2.publicKey, message, signature)).toBe(false);
    });

    it('rejects signature for different message', () => {
      const { publicKey, privateKey } = generateNodeKeyPair();
      const message1 = new TextEncoder().encode('Hello, World!');
      const message2 = new TextEncoder().encode('Goodbye, World!');
      const signature = sign(privateKey, message1);

      expect(verify(publicKey, message2, signature)).toBe(false);
    });
  });

  describe('createAttestation / verifyAttestation', () => {
    it('creates valid attestation', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id);

      expect(attestation.version).toBe(1);
      expect(attestation.principal).toBe(id.principal);
      expect(attestation.domain).toBe('snap2p-nodekey-attestation-v1');
      expect(attestation.signature).toBeInstanceOf(Uint8Array);
      expect(attestation.nonce).toBeInstanceOf(Uint8Array);
      expect(attestation.expiresAt).toBeGreaterThan(attestation.issuedAt);
    });

    it('verifies valid attestation', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id);

      expect(verifyAttestation(attestation, true)).toBe(true);
    });

    it('rejects expired attestation', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id, 1); // 1 second validity

      // Manually expire it
      attestation.expiresAt = Math.floor(Date.now() / 1000) - 600;

      expect(verifyAttestation(attestation, true)).toBe(false);
    });

    it('rejects attestation with tampered signature', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id);

      attestation.signature[0] ^= 0xff;

      expect(verifyAttestation(attestation, true)).toBe(false);
    });

    it('rejects attestation with tampered principal', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id);

      attestation.principal = 'stacks:STfake';

      expect(verifyAttestation(attestation, true)).toBe(false);
    });

    it('respects custom validity period', async () => {
      const id = await createIdentity(true);
      const attestation = await createAttestation(id, 3600); // 1 hour

      const expectedExpiry = attestation.issuedAt + 3600;
      expect(attestation.expiresAt).toBe(expectedExpiry);
    });
  });
});
