/**
 * Gateway Infrastructure Tests (Phase 1)
 *
 * Tests for gateway configuration, IdentityManager, and per-identity storage isolation
 */

import { describe, it, expect, beforeEach, afterEach } from 'vitest';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { createIdentity, saveIdentity } from '../identity/keys.js';
import {
  isGatewayMode,
  loadGatewayConfig,
  saveGatewayConfig,
  validateGatewayConfig,
  createInitialGatewayConfig,
} from '../daemon/gateway-config.js';
import { IdentityManager } from '../daemon/identity-manager.js';
import { GatewayConfigError } from '../types/gateway.js';
import type { GatewayConfig, IdentityConfig } from '../types/gateway.js';

// Test data directory
let testDataDir: string;
let originalHome: string | undefined;

beforeEach(() => {
  // Create temporary test directory
  testDataDir = fs.mkdtempSync(path.join(os.tmpdir(), 'clawchat-test-'));
  originalHome = process.env.HOME;
  process.env.HOME = testDataDir;
});

afterEach(() => {
  // Restore original HOME
  if (originalHome !== undefined) {
    process.env.HOME = originalHome;
  } else {
    delete process.env.HOME;
  }

  // Clean up test directory
  if (testDataDir && fs.existsSync(testDataDir)) {
    fs.rmSync(testDataDir, { recursive: true, force: true });
  }
});

describe('Gateway Configuration', () => {
  describe('isGatewayMode', () => {
    it('returns false when no gateway-config.json exists', () => {
      expect(isGatewayMode()).toBe(false);
    });

    it('returns true when gateway-config.json exists', () => {
      const config = createInitialGatewayConfig('stacks:ST1ABC', 'test', 9000);
      saveGatewayConfig(config);
      expect(isGatewayMode()).toBe(true);
    });
  });

  describe('validateGatewayConfig', () => {
    it('accepts valid configuration', () => {
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            nick: 'test',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).not.toThrow();
    });

    it('rejects invalid version', () => {
      const config = {
        version: 2,
        p2pPort: 9000,
        identities: [],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('Invalid config version');
    });

    it('rejects invalid p2pPort', () => {
      const config = {
        version: 1,
        p2pPort: 99999,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('Invalid p2pPort');
    });

    it('rejects empty identities array', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('At least one identity');
    });

    it('rejects duplicate principals', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
          {
            principal: 'stacks:ST1ABC',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('Duplicate principal');
    });

    it('rejects duplicate nicknames', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
          {
            principal: 'stacks:ST2DEF',
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('Duplicate nickname');
    });

    it('rejects invalid principal format', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'invalid-principal',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('must start with "stacks:"');
    });

    it('rejects invalid nickname characters', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            nick: 'alice@bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('letters, numbers, underscores, and hyphens');
    });

    it('rejects invalid allowedRemotePeers', () => {
      const config = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ABC',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['invalid-peer'],
            openclawWake: true,
          },
        ],
      };

      expect(() => validateGatewayConfig(config)).toThrow(GatewayConfigError);
      expect(() => validateGatewayConfig(config)).toThrow('must be "*" or a principal');
    });
  });

  describe('saveGatewayConfig and loadGatewayConfig', () => {
    it('saves and loads configuration correctly', () => {
      const config = createInitialGatewayConfig('stacks:ST1ABC', 'test', 9000);
      saveGatewayConfig(config);

      const loaded = loadGatewayConfig();
      expect(loaded).toEqual(config);
    });

    it('throws when loading non-existent config', () => {
      expect(() => loadGatewayConfig()).toThrow(GatewayConfigError);
      expect(() => loadGatewayConfig()).toThrow('Gateway config not found');
    });
  });
});

describe('IdentityManager', () => {
  let manager: IdentityManager;

  beforeEach(() => {
    manager = new IdentityManager();
  });

  describe('loadIdentity', () => {
    it('loads identity from per-identity directory', async () => {
      // Create test identity
      const identity = await createIdentity(true);
      const principal = identity.principal;

      // Save in per-identity directory
      const identityDir = path.join(testDataDir, '.clawchat', 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      // Save identity file
      const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
      try {
        process.env.CLAWCHAT_DATA_DIR = identityDir;
        saveIdentity(identity, 'test-password-123');
      } finally {
        if (originalDataDir !== undefined) {
          process.env.CLAWCHAT_DATA_DIR = originalDataDir;
        } else {
          delete process.env.CLAWCHAT_DATA_DIR;
        }
      }

      // Create identity config
      const config: IdentityConfig = {
        principal,
        nick: 'test',
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      // Load via IdentityManager
      await manager.loadIdentity(principal, 'test-password-123', config);

      // Verify loaded
      expect(manager.isLoaded(principal)).toBe(true);
      expect(manager.isLoaded('test')).toBe(true);

      const loaded = manager.getIdentity(principal);
      expect(loaded).not.toBeNull();
      expect(loaded!.identity.principal).toBe(principal);
      expect(loaded!.config.nick).toBe('test');
      expect(loaded!.inbox).toEqual([]);
      expect(loaded!.outbox).toEqual([]);
      expect(loaded!.peers).toEqual([]);
    });

    it('throws when identity directory does not exist', async () => {
      const config: IdentityConfig = {
        principal: 'stacks:ST1NONEXISTENT',
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      await expect(
        manager.loadIdentity('stacks:ST1NONEXISTENT', 'password', config)
      ).rejects.toThrow(GatewayConfigError);
    });

    it('throws when identity is already loaded', async () => {
      // Create and save identity
      const identity = await createIdentity(true);
      const principal = identity.principal;
      const identityDir = path.join(testDataDir, '.clawchat', 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
      try {
        process.env.CLAWCHAT_DATA_DIR = identityDir;
        saveIdentity(identity, 'test-password-123');
      } finally {
        if (originalDataDir !== undefined) {
          process.env.CLAWCHAT_DATA_DIR = originalDataDir;
        } else {
          delete process.env.CLAWCHAT_DATA_DIR;
        }
      }

      const config: IdentityConfig = {
        principal,
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      // Load first time
      await manager.loadIdentity(principal, 'test-password-123', config);

      // Try loading again
      await expect(
        manager.loadIdentity(principal, 'test-password-123', config)
      ).rejects.toThrow(GatewayConfigError);
      await expect(
        manager.loadIdentity(principal, 'test-password-123', config)
      ).rejects.toThrow('already loaded');
    });

    it('throws when nickname conflicts', async () => {
      // Create and save two identities
      const identity1 = await createIdentity(true);
      const identity2 = await createIdentity(true);

      for (const identity of [identity1, identity2]) {
        const identityDir = path.join(
          testDataDir,
          '.clawchat',
          'identities',
          identity.principal
        );
        fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

        const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
        try {
          process.env.CLAWCHAT_DATA_DIR = identityDir;
          saveIdentity(identity, 'test-password-123');
        } finally {
          if (originalDataDir !== undefined) {
            process.env.CLAWCHAT_DATA_DIR = originalDataDir;
          } else {
            delete process.env.CLAWCHAT_DATA_DIR;
          }
        }
      }

      const config1: IdentityConfig = {
        principal: identity1.principal,
        nick: 'alice',
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      const config2: IdentityConfig = {
        principal: identity2.principal,
        nick: 'alice', // Same nickname!
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      // Load first identity
      await manager.loadIdentity(identity1.principal, 'test-password-123', config1);

      // Try loading second with same nickname
      await expect(
        manager.loadIdentity(identity2.principal, 'test-password-123', config2)
      ).rejects.toThrow(GatewayConfigError);
      await expect(
        manager.loadIdentity(identity2.principal, 'test-password-123', config2)
      ).rejects.toThrow('already in use');
    });
  });

  describe('unloadIdentity', () => {
    it('unloads identity and saves state', async () => {
      // Create and load identity
      const identity = await createIdentity(true);
      const principal = identity.principal;
      const identityDir = path.join(testDataDir, '.clawchat', 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
      try {
        process.env.CLAWCHAT_DATA_DIR = identityDir;
        saveIdentity(identity, 'test-password-123');
      } finally {
        if (originalDataDir !== undefined) {
          process.env.CLAWCHAT_DATA_DIR = originalDataDir;
        } else {
          delete process.env.CLAWCHAT_DATA_DIR;
        }
      }

      const config: IdentityConfig = {
        principal,
        nick: 'test',
        autoload: true,
        allowLocal: true,
        allowedRemotePeers: ['*'],
        openclawWake: true,
      };

      await manager.loadIdentity(principal, 'test-password-123', config);
      expect(manager.isLoaded(principal)).toBe(true);

      // Unload
      manager.unloadIdentity(principal);
      expect(manager.isLoaded(principal)).toBe(false);
      expect(manager.isLoaded('test')).toBe(false);
    });
  });

  describe('per-identity storage', () => {
    it('isolates inbox, outbox, and peers per identity', async () => {
      // Create two identities
      const identity1 = await createIdentity(true);
      const identity2 = await createIdentity(true);

      // Save both identities
      for (const identity of [identity1, identity2]) {
        const identityDir = path.join(
          testDataDir,
          '.clawchat',
          'identities',
          identity.principal
        );
        fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

        const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
        try {
          process.env.CLAWCHAT_DATA_DIR = identityDir;
          saveIdentity(identity, 'test-password-123');
        } finally {
          if (originalDataDir !== undefined) {
            process.env.CLAWCHAT_DATA_DIR = originalDataDir;
          } else {
            delete process.env.CLAWCHAT_DATA_DIR;
          }
        }
      }

      // Load both identities
      await manager.loadIdentity(
        identity1.principal,
        'test-password-123',
        {
          principal: identity1.principal,
          nick: 'alice',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: true,
        }
      );

      await manager.loadIdentity(
        identity2.principal,
        'test-password-123',
        {
          principal: identity2.principal,
          nick: 'bob',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: true,
        }
      );

      // Add messages to identity1's inbox
      manager.addToInbox(identity1.principal, {
        id: 'msg1',
        from: identity2.principal,
        to: identity1.principal,
        content: 'Hello Alice',
        timestamp: Date.now(),
        status: 'delivered',
      });

      // Add messages to identity2's inbox
      manager.addToInbox(identity2.principal, {
        id: 'msg2',
        from: identity1.principal,
        to: identity2.principal,
        content: 'Hello Bob',
        timestamp: Date.now(),
        status: 'delivered',
      });

      // Verify isolation
      const loaded1 = manager.getIdentity('alice');
      const loaded2 = manager.getIdentity('bob');

      expect(loaded1!.inbox.length).toBe(1);
      expect(loaded1!.inbox[0].content).toBe('Hello Alice');

      expect(loaded2!.inbox.length).toBe(1);
      expect(loaded2!.inbox[0].content).toBe('Hello Bob');

      // Save and verify persistence
      manager.saveInbox(identity1.principal);
      manager.saveInbox(identity2.principal);

      const inbox1Path = path.join(loaded1!.dataDir, 'inbox.json');
      const inbox2Path = path.join(loaded2!.dataDir, 'inbox.json');

      expect(fs.existsSync(inbox1Path)).toBe(true);
      expect(fs.existsSync(inbox2Path)).toBe(true);

      const inbox1Data = JSON.parse(fs.readFileSync(inbox1Path, 'utf-8'));
      const inbox2Data = JSON.parse(fs.readFileSync(inbox2Path, 'utf-8'));

      expect(inbox1Data[0].content).toBe('Hello Alice');
      expect(inbox2Data[0].content).toBe('Hello Bob');
    });
  });

  describe('getIdentity', () => {
    it('retrieves identity by principal', async () => {
      const identity = await createIdentity(true);
      const principal = identity.principal;
      const identityDir = path.join(testDataDir, '.clawchat', 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
      try {
        process.env.CLAWCHAT_DATA_DIR = identityDir;
        saveIdentity(identity, 'test-password-123');
      } finally {
        if (originalDataDir !== undefined) {
          process.env.CLAWCHAT_DATA_DIR = originalDataDir;
        } else {
          delete process.env.CLAWCHAT_DATA_DIR;
        }
      }

      await manager.loadIdentity(
        principal,
        'test-password-123',
        {
          principal,
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: true,
        }
      );

      const loaded = manager.getIdentity(principal);
      expect(loaded).not.toBeNull();
      expect(loaded!.identity.principal).toBe(principal);
    });

    it('retrieves identity by nickname', async () => {
      const identity = await createIdentity(true);
      const principal = identity.principal;
      const identityDir = path.join(testDataDir, '.clawchat', 'identities', principal);
      fs.mkdirSync(identityDir, { recursive: true, mode: 0o700 });

      const originalDataDir = process.env.CLAWCHAT_DATA_DIR;
      try {
        process.env.CLAWCHAT_DATA_DIR = identityDir;
        saveIdentity(identity, 'test-password-123');
      } finally {
        if (originalDataDir !== undefined) {
          process.env.CLAWCHAT_DATA_DIR = originalDataDir;
        } else {
          delete process.env.CLAWCHAT_DATA_DIR;
        }
      }

      await manager.loadIdentity(
        principal,
        'test-password-123',
        {
          principal,
          nick: 'alice',
          autoload: true,
          allowLocal: true,
          allowedRemotePeers: ['*'],
          openclawWake: true,
        }
      );

      const loaded = manager.getIdentity('alice');
      expect(loaded).not.toBeNull();
      expect(loaded!.identity.principal).toBe(principal);
    });

    it('returns null for non-existent identity', () => {
      const loaded = manager.getIdentity('nonexistent');
      expect(loaded).toBeNull();
    });
  });
});
