/**
 * Gateway E2E Tests
 *
 * End-to-end tests for gateway mode with multiple identities.
 * Tests the full stack: daemon, message routing, ACL, SNaP2P, CLI integration.
 */

import { describe, it, expect, beforeEach, afterEach } from 'vitest';
import * as path from 'path';
import * as fs from 'fs';
import { Daemon } from '../daemon/server.js';
import { createIdentity, saveIdentity, getDataDir, setDataDir } from '../identity/keys.js';
import type { GatewayConfig } from '../types/gateway.js';
import type { Message } from '../types.js';

describe('Gateway E2E Tests', () => {
  let testDir: string;
  let daemon: Daemon;
  let originalDataDir: string;

  beforeEach(async () => {
    // Save original data dir
    originalDataDir = getDataDir();

    // Create temporary test directory
    testDir = path.join('/tmp', `clawchat-test-${Date.now()}`);
    fs.mkdirSync(testDir, { recursive: true, mode: 0o700 });

    // Override data directory for tests
    setDataDir(testDir);
  });

  afterEach(async () => {
    // Clean up
    if (daemon) {
      await daemon.stop(false); // Don't call process.exit
    }

    // Restore original data dir
    setDataDir(originalDataDir);

    if (testDir && fs.existsSync(testDir)) {
      fs.rmSync(testDir, { recursive: true, force: true });
    }
  });

  describe('Multi-Identity Setup', () => {
    it('loads multiple identities on daemon start', async () => {
      // Create two test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Save identities to per-identity directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: [alice.principal],
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      // Load identities manually (simulating what daemon start does)
      const identityManager = (daemon as any).identityManager;
      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Verify both identities are loaded
      const aliceIdentity = identityManager.getIdentity('alice');
      const bobIdentity = identityManager.getIdentity('bob');

      expect(aliceIdentity).not.toBeNull();
      expect(bobIdentity).not.toBeNull();
      expect(aliceIdentity!.identity.principal).toBe(alice.principal);
      expect(bobIdentity!.identity.principal).toBe(bob.principal);
    });
  });

  describe('Message Routing', () => {
    it('routes messages to correct identity inbox', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Send message from Alice to Bob
      const message: Message = {
        id: 'test-message-1',
        from: alice.principal,
        fromNick: 'alice',
        to: bob.principal,
        content: 'Hello Bob!',
        timestamp: Date.now(),
        status: 'delivered',
      };

      identityManager.addToInbox(bob.principal, message);

      // Verify message is in Bob's inbox only
      const bobIdentity = identityManager.getIdentity('bob');
      const aliceIdentity = identityManager.getIdentity('alice');

      expect(bobIdentity!.inbox).toHaveLength(1);
      expect(bobIdentity!.inbox[0].content).toBe('Hello Bob!');
      expect(aliceIdentity!.inbox).toHaveLength(0);
    });

    it('isolates outbox per identity', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Add message to Alice's outbox
      const aliceMessage: Message = {
        id: 'test-message-2',
        from: alice.principal,
        fromNick: 'alice',
        to: 'stacks:ST3CHARLIE',
        content: 'Hello Charlie from Alice!',
        timestamp: Date.now(),
        status: 'pending',
      };

      identityManager.addToOutbox(alice.principal, aliceMessage);

      // Add message to Bob's outbox
      const bobMessage: Message = {
        id: 'test-message-3',
        from: bob.principal,
        fromNick: 'bob',
        to: 'stacks:ST4DAVE',
        content: 'Hello Dave from Bob!',
        timestamp: Date.now(),
        status: 'pending',
      };

      identityManager.addToOutbox(bob.principal, bobMessage);

      // Verify outboxes are isolated
      const aliceIdentity = identityManager.getIdentity('alice');
      const bobIdentity = identityManager.getIdentity('bob');

      expect(aliceIdentity!.outbox).toHaveLength(1);
      expect(aliceIdentity!.outbox[0].content).toBe('Hello Charlie from Alice!');

      expect(bobIdentity!.outbox).toHaveLength(1);
      expect(bobIdentity!.outbox[0].content).toBe('Hello Dave from Bob!');
    });
  });

  describe('ACL Enforcement', () => {
    it('accepts message from allowed peer', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config - Bob only allows Alice
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: [alice.principal], // Only allow Alice
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      const messageRouter = (daemon as any).messageRouter;

      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Message from Alice to Bob (should succeed)
      const aliceMessage: Message = {
        id: 'test-message-4',
        from: alice.principal,
        to: bob.principal,
        content: 'Hello Bob!',
        timestamp: Date.now(),
        status: 'delivered',
      };

      const result = messageRouter.routeInbound(aliceMessage, alice.principal);
      expect(result.success).toBe(true);
      expect(result.identity?.identity.principal).toBe(bob.principal);
    });

    it('rejects message from non-allowed peer', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config - Bob only allows specific peer (not Alice)
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['stacks:ST3CHARLIE'], // Only allow Charlie, not Alice
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      const messageRouter = (daemon as any).messageRouter;

      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Message from Alice to Bob (should fail - Alice not in Bob's ACL)
      const aliceMessage: Message = {
        id: 'test-message-5',
        from: alice.principal,
        to: bob.principal,
        content: 'Hello Bob!',
        timestamp: Date.now(),
        status: 'delivered',
      };

      const result = messageRouter.routeInbound(aliceMessage, alice.principal);
      expect(result.success).toBe(false);
      expect(result.error).toContain('not in allowedRemotePeers');
    });

    it('allows wildcard ACL to accept any peer', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';

      // Set up directory
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config - Alice allows all peers (*)
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'], // Allow all
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      const messageRouter = (daemon as any).messageRouter;

      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);

      // Message from random peer to Alice (should succeed due to wildcard)
      const randomMessage: Message = {
        id: 'test-message-6',
        from: 'stacks:ST3RANDOM',
        to: alice.principal,
        content: 'Hello Alice!',
        timestamp: Date.now(),
        status: 'delivered',
      };

      const result = messageRouter.routeInbound(randomMessage, 'stacks:ST3RANDOM');
      expect(result.success).toBe(true);
      expect(result.identity?.identity.principal).toBe(alice.principal);
    });
  });

  describe('Identity Resolution', () => {
    it('uses default identity when --as not specified', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config - Alice first, Bob second
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;
      const messageRouter = (daemon as any).messageRouter;

      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Get default identity (should be first loaded = Alice)
      const defaultIdentity = messageRouter.getDefaultIdentity();
      expect(defaultIdentity).not.toBeNull();
      expect(defaultIdentity!.identity.principal).toBe(alice.principal);
      expect(defaultIdentity!.identity.nick).toBe('alice');
    });

    it('resolves identity by nickname', async () => {
      // Create test identities
      const alice = await createIdentity(true);
      alice.nick = 'alice';
      const bob = await createIdentity(true);
      bob.nick = 'bob';

      // Set up directories
      const aliceDir = path.join(testDir, 'identities', alice.principal);
      const bobDir = path.join(testDir, 'identities', bob.principal);

      fs.mkdirSync(aliceDir, { recursive: true, mode: 0o700 });
      fs.mkdirSync(bobDir, { recursive: true, mode: 0o700 });

      process.env.CLAWCHAT_DATA_DIR = aliceDir;
      saveIdentity(alice, 'password1234');
      fs.writeFileSync(path.join(aliceDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(aliceDir, 'peers.json'), '[]', { mode: 0o600 });

      process.env.CLAWCHAT_DATA_DIR = bobDir;
      saveIdentity(bob, 'password1234');
      fs.writeFileSync(path.join(bobDir, 'inbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'outbox.json'), '[]', { mode: 0o600 });
      fs.writeFileSync(path.join(bobDir, 'peers.json'), '[]', { mode: 0o600 });

      delete process.env.CLAWCHAT_DATA_DIR;

      // Create gateway config
      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: alice.principal,
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
          {
            principal: bob.principal,
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false,
          },
        ],
      };

      // Create daemon
      daemon = new Daemon({
        p2pPort: 9000,
        gatewayConfig: config,
      });

      const identityManager = (daemon as any).identityManager;

      await identityManager.loadIdentity(alice.principal, 'password1234', config.identities[0]);
      await identityManager.loadIdentity(bob.principal, 'password1234', config.identities[1]);

      // Resolve by nickname
      const bobIdentity = identityManager.getIdentity('bob');
      expect(bobIdentity).not.toBeNull();
      expect(bobIdentity!.identity.principal).toBe(bob.principal);
      expect(bobIdentity!.identity.nick).toBe('bob');
    });
  });

  describe('OpenClaw Wake Configuration', () => {
    it('respects per-identity openclawWake setting', () => {
      // This test verifies the configuration structure
      // Actual wake functionality is tested in daemon.test.ts

      const config: GatewayConfig = {
        version: 1,
        p2pPort: 9000,
        identities: [
          {
            principal: 'stacks:ST1ALICE',
            nick: 'alice',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: true, // Wake enabled for Alice
          },
          {
            principal: 'stacks:ST2BOB',
            nick: 'bob',
            autoload: true,
            allowLocal: true,
            allowedRemotePeers: ['*'],
            openclawWake: false, // Wake disabled for Bob
          },
        ],
      };

      // Verify configuration
      expect(config.identities[0].openclawWake).toBe(true);
      expect(config.identities[1].openclawWake).toBe(false);
    });
  });
});
