import { describe, it, expect } from 'vitest';
import {
  generateX25519KeyPair,
  initHandshake,
  initiatorMsg1,
  responderMsg2,
  initiatorMsg3,
  responderFinalize,
  deriveSessionKeys,
  encrypt,
  decrypt,
} from '../crypto/noise.js';

describe('Crypto', () => {
  describe('generateX25519KeyPair', () => {
    it('generates valid X25519 keypair', () => {
      const kp = generateX25519KeyPair();

      expect(kp.publicKey).toBeInstanceOf(Uint8Array);
      expect(kp.privateKey).toBeInstanceOf(Uint8Array);
      expect(kp.publicKey.length).toBe(32);
      expect(kp.privateKey.length).toBe(32);
    });

    it('generates unique keypairs', () => {
      const kp1 = generateX25519KeyPair();
      const kp2 = generateX25519KeyPair();

      expect(Buffer.from(kp1.publicKey).toString('hex'))
        .not.toBe(Buffer.from(kp2.publicKey).toString('hex'));
    });
  });

  describe('Noise XX Handshake', () => {
    it('completes full handshake between initiator and responder', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      // Initialize both sides
      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);

      // -> e (initiator sends ephemeral)
      const msg1 = initiatorMsg1(initiatorState);
      expect(msg1.length).toBe(32);

      // <- e, ee, s, es (responder processes and responds)
      const msg2 = responderMsg2(responderState, msg1);
      expect(msg2.length).toBe(64); // ephemeral + static

      // -> s, se (initiator processes and responds)
      const msg3 = initiatorMsg3(initiatorState, msg2);
      expect(msg3.length).toBe(32); // static

      // Responder finalizes
      responderFinalize(responderState, msg3);

      // Both sides should have remote static keys
      expect(initiatorState.remoteStaticPub).toBeDefined();
      expect(responderState.remoteStaticPub).toBeDefined();

      // Remote static should match the other's local static
      expect(Buffer.from(initiatorState.remoteStaticPub!).toString('hex'))
        .toBe(Buffer.from(responderStatic.publicKey).toString('hex'));
      expect(Buffer.from(responderState.remoteStaticPub!).toString('hex'))
        .toBe(Buffer.from(initiatorStatic.publicKey).toString('hex'));
    });

    it('derives matching session keys', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);

      const msg1 = initiatorMsg1(initiatorState);
      const msg2 = responderMsg2(responderState, msg1);
      const msg3 = initiatorMsg3(initiatorState, msg2);
      responderFinalize(responderState, msg3);

      const initiatorKeys = deriveSessionKeys(initiatorState, true);
      const responderKeys = deriveSessionKeys(responderState, false);

      // Initiator's send key should match responder's recv key
      expect(Buffer.from(initiatorKeys.sendCipher.key).toString('hex'))
        .toBe(Buffer.from(responderKeys.recvCipher.key).toString('hex'));

      // Initiator's recv key should match responder's send key
      expect(Buffer.from(initiatorKeys.recvCipher.key).toString('hex'))
        .toBe(Buffer.from(responderKeys.sendCipher.key).toString('hex'));

      // Handshake hashes should match
      expect(Buffer.from(initiatorKeys.handshakeHash).toString('hex'))
        .toBe(Buffer.from(responderKeys.handshakeHash).toString('hex'));
    });
  });

  describe('ChaCha20-Poly1305 Encryption', () => {
    it('encrypts and decrypts message', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      // Complete handshake
      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);
      const msg1 = initiatorMsg1(initiatorState);
      const msg2 = responderMsg2(responderState, msg1);
      const msg3 = initiatorMsg3(initiatorState, msg2);
      responderFinalize(responderState, msg3);

      const initiatorKeys = deriveSessionKeys(initiatorState, true);
      const responderKeys = deriveSessionKeys(responderState, false);

      // Initiator sends message
      const plaintext = new TextEncoder().encode('Hello, secure world!');
      const ciphertext = encrypt(initiatorKeys.sendCipher, plaintext);

      expect(ciphertext.length).toBe(plaintext.length + 16); // +16 for auth tag

      // Responder decrypts
      const decrypted = decrypt(responderKeys.recvCipher, ciphertext);

      expect(new TextDecoder().decode(decrypted)).toBe('Hello, secure world!');
    });

    it('handles multiple messages with nonce increment', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);
      const msg1 = initiatorMsg1(initiatorState);
      const msg2 = responderMsg2(responderState, msg1);
      const msg3 = initiatorMsg3(initiatorState, msg2);
      responderFinalize(responderState, msg3);

      const initiatorKeys = deriveSessionKeys(initiatorState, true);
      const responderKeys = deriveSessionKeys(responderState, false);

      const messages = ['First message', 'Second message', 'Third message'];

      for (const msg of messages) {
        const plaintext = new TextEncoder().encode(msg);
        const ciphertext = encrypt(initiatorKeys.sendCipher, plaintext);
        const decrypted = decrypt(responderKeys.recvCipher, ciphertext);
        expect(new TextDecoder().decode(decrypted)).toBe(msg);
      }
    });

    it('produces different ciphertext for same plaintext', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);
      const msg1 = initiatorMsg1(initiatorState);
      const msg2 = responderMsg2(responderState, msg1);
      const msg3 = initiatorMsg3(initiatorState, msg2);
      responderFinalize(responderState, msg3);

      const initiatorKeys = deriveSessionKeys(initiatorState, true);

      const plaintext = new TextEncoder().encode('Same message');
      const ciphertext1 = encrypt(initiatorKeys.sendCipher, plaintext);
      const ciphertext2 = encrypt(initiatorKeys.sendCipher, plaintext);

      // Different due to nonce increment
      expect(Buffer.from(ciphertext1).toString('hex'))
        .not.toBe(Buffer.from(ciphertext2).toString('hex'));
    });

    it('rejects tampered ciphertext', () => {
      const initiatorStatic = generateX25519KeyPair();
      const responderStatic = generateX25519KeyPair();

      const initiatorState = initHandshake(initiatorStatic, true);
      const responderState = initHandshake(responderStatic, false);
      const msg1 = initiatorMsg1(initiatorState);
      const msg2 = responderMsg2(responderState, msg1);
      const msg3 = initiatorMsg3(initiatorState, msg2);
      responderFinalize(responderState, msg3);

      const initiatorKeys = deriveSessionKeys(initiatorState, true);
      const responderKeys = deriveSessionKeys(responderState, false);

      const plaintext = new TextEncoder().encode('Secret message');
      const ciphertext = encrypt(initiatorKeys.sendCipher, plaintext);

      // Tamper with ciphertext
      ciphertext[5] ^= 0xff;

      expect(() => decrypt(responderKeys.recvCipher, ciphertext)).toThrow();
    });
  });
});
