#!/bin/bash
# EXAMPLE: Multi-agent ClawChat setup for coordinated systems
# 
# This example demonstrates setting up multiple agents that need to communicate.
# Customize the agent names, ports, and passwords for your specific use case.
#
# WARNING: This example uses simple passwords for demonstration only!
# In production:
# - Use strong, unique passwords
# - Store credentials securely (e.g., environment variables, secret management)
# - Never commit passwords to version control

set -e

echo "=== ClawChat Multi-Agent Setup Example ==="
echo ""
echo "This script demonstrates setting up multiple coordinating agents."
echo "Modify it for your specific use case!"
echo ""

# CUSTOMIZE THESE: Define your agent configurations
# Example shows a coordinator pattern with worker agents
declare -A AGENTS=(
    ["coordinator"]="9100"    # Main coordinating agent
    ["worker1"]="9101"        # Worker agent 1
    ["worker2"]="9102"        # Worker agent 2
    # Add more agents as needed for your use case
)

# SECURITY WARNING: Example uses predictable passwords - DO NOT USE IN PRODUCTION
# Consider using:
# - Environment variables: PASSWORD="${AGENT_NAME}_PASSWORD"
# - Password manager integration
# - Secure key management service

# Colors for output
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

echo -e "${RED}⚠️  SECURITY WARNING: This example uses simple passwords!${NC}"
echo -e "${RED}   For production, implement proper credential management.${NC}"
echo ""
read -p "Continue with example setup? (y/n) " -n 1 -r
echo ""
if [[ ! $REPLY =~ ^[Yy]$ ]]; then
    exit 1
fi

# Function to create agent identity
create_agent() {
    local name=$1
    local port=$2
    local data_dir="$HOME/.clawchat-$name"
    
    echo -e "${YELLOW}Setting up $name on port $port...${NC}"
    
    # Check if identity already exists
    if [ -f "$data_dir/identity.enc" ]; then
        echo "  ✓ Identity already exists for $name"
        # Show the principal (example password - customize this!)
        local principal=$(clawchat --data-dir "$data_dir" identity show --password "example-${name}-password" 2>/dev/null | jq -r '.principal' || echo "unknown")
        echo "    Principal: $principal"
    else
        echo "  Creating new identity for $name..."
        # EXAMPLE: Using predictable password - CHANGE THIS!
        local password="example-${name}-password"
        
        local result=$(clawchat --data-dir "$data_dir" identity create --password "$password")
        local principal=$(echo "$result" | jq -r '.principal')
        local mnemonic=$(echo "$result" | jq -r '.mnemonic')
        
        # Set nickname
        clawchat --data-dir "$data_dir" identity set-nick "$name" --password "$password" >/dev/null
        
        echo "  ✓ Created identity: $principal"
        echo ""
        echo -e "  ${RED}⚠️  SAVE THIS SEED PHRASE for $name:${NC}"
        echo "  $mnemonic"
        echo ""
        
        # Example only - DO NOT save credentials to files in production!
        echo "  (In production, store these securely, not in plain text files!)"
    fi
    
    echo ""
}

# Create all identities
echo -e "${GREEN}Step 1: Creating identities...${NC}"
for agent in "${!AGENTS[@]}"; do
    create_agent "$agent" "${AGENTS[$agent]}"
done

echo ""
echo -e "${GREEN}=== Example Setup Complete ===${NC}"
echo ""
echo "To complete setup for your use case:"
echo "1. Replace example agent names with your actual agent names"
echo "2. Use secure passwords (not the example ones)"
echo "3. Implement proper credential management"
echo "4. Start daemons with: clawchat daemon start --port PORT"
echo "5. Connect agents by exchanging their principals"
echo ""
echo "See the skill documentation for production deployment patterns."
