#!/usr/bin/env node
/**
 * claw402 x402 query script
 * Usage: node query.mjs <path> [key=value ...]
 * Example: node query.mjs /api/v1/nofx/netflow/top-ranking limit=20 duration=1h
 */

import { wrapFetchWithPayment } from 'x402-fetch'
import { privateKeyToAccount } from 'viem/accounts'

const [,, path, ...paramArgs] = process.argv

if (!path) {
  console.error(JSON.stringify({ error: 'Usage: node query.mjs <path> [key=value ...]' }))
  process.exit(1)
}

const privateKey = process.env.WALLET_PRIVATE_KEY
if (!privateKey) {
  console.error(JSON.stringify({ error: 'WALLET_PRIVATE_KEY environment variable is required' }))
  process.exit(1)
}

const GATEWAY = process.env.CLAW402_GATEWAY ?? 'https://claw402.ai'

// Parse key=value params
const params = {}
for (const arg of paramArgs) {
  const idx = arg.indexOf('=')
  if (idx > 0) params[arg.slice(0, idx)] = arg.slice(idx + 1)
}

const qs = new URLSearchParams(params).toString()
const url = `${GATEWAY}${path}${qs ? '?' + qs : ''}`

try {
  const account = privateKeyToAccount(/** @type {`0x${string}`} */ (privateKey))
  const fetchWithPayment = wrapFetchWithPayment(fetch, { wallet: account })

  const res = await fetchWithPayment(url)
  const data = await res.json()

  console.log(JSON.stringify({ status: res.status, url, data }, null, 2))
} catch (err) {
  console.error(JSON.stringify({ error: String(err), url }))
  process.exit(1)
}
